/*
 * Copyright 2013 Canonical Ltd.
 * Copyright 2023 Robert Tari
 *
 * Authors:
 *   Charles Kerr <charles.kerr@canonical.com>
 *   Robert Tari <robert@tari.in>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <gtest/gtest.h>

#include "device.h"
#include "service.h"

/***
****
***/

namespace
{
  void quiet_log_func (const gchar *log_domain,
                       GLogLevelFlags log_level,
                       const gchar *message,
                       gpointer user_data)
  {
    // instantiating an indicator w/o a window causes lots
    // of glib/gtk warnings... silence them so that they don't
    // obscure any other warnings generated by the tests.
  }

  void ensure_glib_initialized ()
  {
    static bool initialized = false;

    if (G_UNLIKELY(!initialized))
    {
      initialized = true;
      g_log_set_handler ("Gtk", (GLogLevelFlags)(G_LOG_LEVEL_CRITICAL|G_LOG_LEVEL_WARNING), quiet_log_func, NULL);
      g_log_set_handler ("GLib-GObject", (GLogLevelFlags)(G_LOG_LEVEL_CRITICAL|G_LOG_LEVEL_WARNING), quiet_log_func, NULL);
    }
  }
}

/***
****
***/

class IndicatorTest : public ::testing::Test
{
  protected:

    IndicatorPowerDevice * ac_device;
    IndicatorPowerDevice * battery_device;

    virtual void SetUp()
    {
      ensure_glib_initialized ();

      g_setenv( "GSETTINGS_SCHEMA_DIR", SCHEMA_DIR, TRUE);

      ac_device = indicator_power_device_new (
        "/org/freedesktop/UPower/devices/line_power_AC",
        UP_DEVICE_KIND_LINE_POWER, "Some Model",
        0.0, UP_DEVICE_STATE_UNKNOWN, 0);

      battery_device = indicator_power_device_new (
        "/org/freedesktop/UPower/devices/battery_BAT0",
        UP_DEVICE_KIND_BATTERY, "Some Model",
        52.871712, UP_DEVICE_STATE_DISCHARGING, 8834);
    }

    virtual void TearDown()
    {
      ASSERT_EQ (1, G_OBJECT(battery_device)->ref_count);
      ASSERT_EQ (1, G_OBJECT(ac_device)->ref_count);
      g_object_unref (battery_device);
      g_object_unref (ac_device);
    }

    const char* GetAccessibleDesc (IndicatorPower * power) const
    {
      GList * entries = indicator_object_get_entries (INDICATOR_OBJECT(power));
      g_assert (g_list_length(entries) == 1);
      IndicatorObjectEntry * entry = static_cast<IndicatorObjectEntry*>(entries->data);
      const char * ret = entry->accessible_desc;
      g_list_free (entries);
      return ret;
    }
};

/***
****
***/

TEST_F(IndicatorTest, GObjectNew)
{
  GObject * o = G_OBJECT (g_object_new (INDICATOR_POWER_TYPE, NULL));
  ASSERT_TRUE (o != NULL);
  ASSERT_TRUE (IS_INDICATOR_POWER(o));
  g_object_run_dispose (o); // used to get coverage of both branches in the object's dispose func's g_clear_*() calls
  g_object_unref (o);
}

TEST_F(IndicatorTest, SetDevices)
{
  GSList * devices;
  IndicatorPower * power = INDICATOR_POWER(g_object_new (INDICATOR_POWER_TYPE, NULL));

  devices = NULL;
  devices = g_slist_append (devices, ac_device);
  devices = g_slist_append (devices, battery_device);
  indicator_power_set_devices (power, devices);
  g_slist_free (devices);

  g_object_unref (power);
}

TEST_F(IndicatorTest, DischargingStrings)
{
  IndicatorPower * power = INDICATOR_POWER(g_object_new (INDICATOR_POWER_TYPE, NULL));
  GSList * devices = g_slist_append (NULL, battery_device);

  // give the indicator a discharging battery with 30 minutes of life left
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_STATE, UP_DEVICE_STATE_DISCHARGING,
                INDICATOR_POWER_DEVICE_PERCENTAGE, 50.0,
                INDICATOR_POWER_DEVICE_TIME, guint64(60*30),
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (30 minutes left (50%))");

  // give the indicator a discharging battery with 1 hour of life left
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_STATE, UP_DEVICE_STATE_DISCHARGING,
                INDICATOR_POWER_DEVICE_PERCENTAGE, 50.0,
                INDICATOR_POWER_DEVICE_TIME, guint64(60*60),
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (1 hour left (50%))");

  // give the indicator a discharging battery with 2 hours of life left
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_PERCENTAGE, 100.0,
                INDICATOR_POWER_DEVICE_TIME, guint64(60*60*2),
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (2 hours left (100%))");

  // give the indicator a discharging battery with over 12 hours of life left
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_TIME, guint64(60*60*12 + 1),
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery");

  // give the indicator a discharging battery with 29 seconds left
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_TIME, guint64(29),
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (Unknown time left (100%))");

  // what happens if the time estimate isn't available
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_TIME, guint64(0),
                INDICATOR_POWER_DEVICE_PERCENTAGE, 50.0,
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (50%)");

  // what happens if the time estimate AND percentage isn't available
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_TIME, guint64(0),
                INDICATOR_POWER_DEVICE_PERCENTAGE, 0.0,
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (not present)");

  // cleanup
  g_slist_free (devices);
  g_object_unref (power);
}

TEST_F(IndicatorTest, ChargingStrings)
{
  IndicatorPower * power = INDICATOR_POWER(g_object_new (INDICATOR_POWER_TYPE, NULL));
  GSList * devices = g_slist_prepend (NULL, battery_device);

  // give the indicator a discharging battery with 1 hour of life left
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_STATE, UP_DEVICE_STATE_CHARGING,
                INDICATOR_POWER_DEVICE_PERCENTAGE, 50.0,
                INDICATOR_POWER_DEVICE_TIME, guint64(60*60),
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (1 hour to charge (50%))");

  // give the indicator a discharging battery with 2 hours of life left
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_TIME, guint64(60*60*2),
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (2 hours to charge (50%))");

  // cleanup
  g_slist_free (devices);
  g_object_unref (power);
}

TEST_F(IndicatorTest, ChargedStrings)
{
  IndicatorPower * power = INDICATOR_POWER(g_object_new (INDICATOR_POWER_TYPE, NULL));
  GSList * devices = g_slist_append (NULL, battery_device);

  // give the indicator a discharging battery with 1 hour of life left
  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_STATE, UP_DEVICE_STATE_FULLY_CHARGED,
                INDICATOR_POWER_DEVICE_PERCENTAGE, 100.0,
                INDICATOR_POWER_DEVICE_TIME, guint64(0),
                NULL);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (charged)");

  // cleanup
  g_slist_free (devices);
  g_object_unref (power);
}

TEST_F(IndicatorTest, AvoidChargingBatteriesWithZeroSecondsLeft)
{
  IndicatorPower * power = INDICATOR_POWER(g_object_new (INDICATOR_POWER_TYPE, NULL));

  g_object_set (battery_device,
                INDICATOR_POWER_DEVICE_STATE, UP_DEVICE_STATE_FULLY_CHARGED,
                INDICATOR_POWER_DEVICE_PERCENTAGE, 100.0,
                INDICATOR_POWER_DEVICE_TIME, guint64(0),
                NULL);
  IndicatorPowerDevice * bad_battery_device  = indicator_power_device_new (
    "/org/freedesktop/UPower/devices/battery_BAT0",
    UP_DEVICE_KIND_BATTERY, "Some Model",
    53, UP_DEVICE_STATE_CHARGING, 0);

  GSList * devices = NULL;
  devices = g_slist_append (devices, battery_device);
  devices = g_slist_append (devices, bad_battery_device);
  indicator_power_set_devices (power, devices);
  ASSERT_STREQ (GetAccessibleDesc(power), "Battery (53%)");

  // cleanup
  g_slist_free (devices);
  g_object_unref (power);
  g_object_unref (bad_battery_device);
}

TEST_F(IndicatorTest, OtherDevices)
{
  IndicatorPower * power = INDICATOR_POWER(g_object_new (INDICATOR_POWER_TYPE, NULL));

  g_object_ref (battery_device);
  GSList * devices = g_slist_append (NULL, battery_device);

  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/mouse", UP_DEVICE_KIND_MOUSE, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/ups", UP_DEVICE_KIND_UPS, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/keyboard", UP_DEVICE_KIND_KEYBOARD, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/pda", UP_DEVICE_KIND_PDA, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/phone", UP_DEVICE_KIND_PHONE, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/monitor", UP_DEVICE_KIND_MONITOR, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/media_player", UP_DEVICE_KIND_MEDIA_PLAYER, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/tablet", UP_DEVICE_KIND_TABLET, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/computer", UP_DEVICE_KIND_COMPUTER, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));
  devices = g_slist_append (devices, indicator_power_device_new (
    "/org/freedesktop/UPower/devices/unknown", UP_DEVICE_KIND_UNKNOWN, "Some Model",
    0, UP_DEVICE_STATE_UNKNOWN, 0));

  indicator_power_set_devices (power, devices);

  // FIXME: this tests to confirm the code doesn't crash,
  // but further tests would be helpful

  // cleanup
  g_slist_free_full (devices, g_object_unref);
  g_object_unref (power);
}

TEST_F(IndicatorTest, NoDevices)
{
  IndicatorPower * power = INDICATOR_POWER(g_object_new (INDICATOR_POWER_TYPE, NULL));

  indicator_power_set_devices (power, NULL);

  // FIXME: this tests to confirm the code doesn't crash,
  // but further tests would be helpful

  // cleanup
  g_object_unref (power);
}
