//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/Interface/LayerRoughness.cpp
//! @brief     Implements class LayerRoughness.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Sample/Interface/LayerRoughness.h"
#include "Base/Py/PyFmt.h"
#include "Base/Util/Assert.h"
#include <boost/math/special_functions/bessel.hpp>
#include <numbers>
using std::numbers::pi;

//! Constructor of layer roughness.
//! @param sigma: rms of the roughness in nanometers
//! @param hurstParameter: hurst parameter which describes how jagged the interface,
//! dimensionless [0.0, 1.0], where 0.0 gives more spikes, 1.0 more smoothness
//! @param lateralCorrLength: lateral correlation length of the roughness in nanometers
LayerRoughness::LayerRoughness(double sigma, double hurstParameter, double lateralCorrLength)
    : m_sigma(sigma)
    , m_hurstParameter(hurstParameter)
    , m_lateralCorrLength(lateralCorrLength)
{
    validateOrThrow();
}

LayerRoughness::LayerRoughness()
    : LayerRoughness(0, 0, 0)
{
}

//! Power spectral density of the surface roughness is a result of two-dimensional
//! Fourier transform of the correlation function of the roughness profile.
//!
//! Based on Palasantzas, Phys Rev B, 48, 14472 (1993)
double LayerRoughness::spectralFunction(const R3 kvec) const
{
    ASSERT(m_validated);
    double H = m_hurstParameter;
    double clength2 = m_lateralCorrLength * m_lateralCorrLength;
    double Qpar2 = kvec.x() * kvec.x() + kvec.y() * kvec.y();
    return 4.0 * pi * H * m_sigma * m_sigma * clength2 * std::pow(1 + Qpar2 * clength2, -1 - H);
}

//! Correlation function of the roughness profile
double LayerRoughness::corrFunction(const R3 k) const
{
    ASSERT(m_validated);
    double H = m_hurstParameter;
    double clength = m_lateralCorrLength;
    double R = sqrt(k.x() * k.x() + k.y() * k.y());
    return m_sigma * m_sigma * std::pow(2., 1 - H) / tgamma(H) * std::pow(R / clength, H)
           * boost::math::cyl_bessel_k(H, R / clength);
}

std::string LayerRoughness::pythonConstructor() const
{
    return Py::Fmt::printFunction(
        "LayerRoughness", {{m_sigma, ""}, {m_hurstParameter, ""}, {m_lateralCorrLength, "nm"}});
}

std::string LayerRoughness::validate() const
{
    std::vector<std::string> errs;
    requestGe0(errs, m_sigma, "sigma");
    if (m_sigma > 0) {
        requestIn(errs, m_hurstParameter, "hurst", 0, 1);
        requestGe0(errs, m_lateralCorrLength, "lateralCorrLength"); // TODO why not gt0 ?
    }
    if (!errs.empty())
        return jointError(errs);
    m_validated = true;
    return "";
}
