/* This file is part of the KDE project
   Copyright (C) 2005,2006 Jarosław Staniek <staniek@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "kexicsvexport.h"
#include "kexicsvwidgets.h"
#include <db/cursor.h>
#include <db/utils.h>
#include <core/KexiMainWindowIface.h>
#include <core/kexiproject.h>
#include <core/kexipartinfo.h>
#include <core/kexipartmanager.h>
#include <core/kexiguimsghandler.h>
#include <kexiutils/utils.h>
#include <widget/kexicharencodingcombobox.h>

#include <QTextStream>
#include <QCheckBox>
#include <QGroupBox>
#include <QClipboard>
#include <kapplication.h>
#include <klocale.h>
#include <kdebug.h>
#include <ksavefile.h>


using namespace KexiCSVExport;

Options::Options()
        : mode(File), itemId(0), addColumnNames(true)
{
}

bool Options::assign(QMap<QString, QString>& args)
{
    mode = (args["destinationType"] == "file")
           ? KexiCSVExport::File : KexiCSVExport::Clipboard;

    if (args.contains("delimiter"))
        delimiter = args["delimiter"];
    else
        delimiter = (mode == File) ? KEXICSV_DEFAULT_FILE_DELIMITER : KEXICSV_DEFAULT_CLIPBOARD_DELIMITER;

    if (args.contains("textQuote"))
        textQuote = args["textQuote"];
    else
        textQuote = (mode == File) ? KEXICSV_DEFAULT_FILE_TEXT_QUOTE : KEXICSV_DEFAULT_CLIPBOARD_TEXT_QUOTE;

    bool ok;
    itemId = args["itemId"].toInt(&ok);
    if (!ok || itemId <= 0)
        return false;
    if (args.contains("forceDelimiter"))
        forceDelimiter = args["forceDelimiter"];
    if (args.contains("addColumnNames"))
        addColumnNames = (args["addColumnNames"] == "1");
    return true;
}

//------------------------------------

bool KexiCSVExport::exportData(KexiDB::TableOrQuerySchema& tableOrQuery,
                               const Options& options, int rowCount, QTextStream *predefinedTextStream)
{
    KexiDB::Connection* conn = tableOrQuery.connection();
    if (!conn)
        return false;

    KexiDB::QuerySchema* query = tableOrQuery.query();
    QList<QVariant> queryParams;
    if (!query) {
        query = tableOrQuery.table()->query();
    }
    else {
        queryParams = KexiMainWindowIface::global()->currentParametersForQuery(query->id());
    }

    if (rowCount == -1)
        rowCount = KexiDB::rowCount(tableOrQuery, queryParams);
    if (rowCount == -1)
        return false;

//! @todo move this to non-GUI location so it can be also used via command line
//! @todo add a "finish" page with a progressbar.
//! @todo look at rowCount whether the data is really large;
//!       if so: avoid copying to clipboard (or ask user) because of system memory

//! @todo OPTIMIZATION: use fieldsExpanded(true /*UNIQUE*/)
//! @todo OPTIMIZATION? (avoid multiple data retrieving) look for already fetched data within KexiProject..

    KexiDB::QueryColumnInfo::Vector fields(query->fieldsExpanded(KexiDB::QuerySchema::WithInternalFields));
    QString buffer;

    QScopedPointer<KSaveFile> kSaveFile;
    QTextStream *stream = 0;
    QScopedPointer<QTextStream> kSaveFileTextStream;

    const bool copyToClipboard = options.mode == Clipboard;
    if (copyToClipboard) {
//! @todo (during exporting): enlarge bufSize by factor of 2 when it became too small
        uint bufSize = qMin(uint(rowCount < 0 ? 10 : rowCount) * fields.count() * 20, (uint)128000);
        buffer.reserve(bufSize);
        if ((uint)buffer.capacity() < bufSize) {
            kWarning() << "Cannot allocate memory for " << bufSize
            << " characters";
            return false;
        }
    } else {
        if (predefinedTextStream) {
            stream = predefinedTextStream;
        } else {
            if (options.fileName.isEmpty()) {//sanity
                kWarning() << "Fname is empty";
                return false;
            }
            kSaveFile.reset(new KSaveFile(options.fileName));

            kDebug() << "KSaveFile Filename:" << kSaveFile->fileName();

            if (kSaveFile->open()) {
                kSaveFileTextStream.reset(new QTextStream(kSaveFile.data()));
                stream = kSaveFileTextStream.data();
                kDebug() << "have a stream";
            }
            if (QFile::NoError != kSaveFile->error() || !stream) {//sanity
                kWarning() << "Status != 0 or stream == 0";
//! @todo show error
                return false;
            }
        }
    }

//! @todo escape strings

#define _ERR \
    if (kSaveFile) { kSaveFile->abort(); } \
    return false

#define APPEND(what) \
    if (copyToClipboard) buffer.append(what); else (*stream) << (what)

// use native line ending for copying, RFC 4180 one for saving to file
#define APPEND_EOLN \
    if (copyToClipboard) { APPEND('\n'); } else { APPEND("\r\n"); }

    kDebug() << 0 << "Columns: " << query->fieldsExpanded().count();
    // 0. Cache information
    const uint fieldsCount = query->fieldsExpanded().count(); //real fields count without internals
    const QByteArray delimiter(options.delimiter.left(1).toLatin1());
    const bool hasTextQuote = !options.textQuote.isEmpty();
    const QString textQuote(options.textQuote.left(1));
    const QByteArray escapedTextQuote((textQuote + textQuote).toLatin1());   //ok?
    //cache for faster checks
    QScopedArrayPointer<bool> isText(new bool[fieldsCount]);
    QScopedArrayPointer<bool> isDateTime(new bool[fieldsCount]);
    QScopedArrayPointer<bool> isTime(new bool[fieldsCount]);
    QScopedArrayPointer<bool> isBLOB(new bool[fieldsCount]);
    QScopedArrayPointer<uint> visibleFieldIndex(new uint[fieldsCount]);
// bool isInteger[fieldsCount]; //cache for faster checks
// bool isFloatingPoint[fieldsCount]; //cache for faster checks
    for (uint i = 0; i < fieldsCount; i++) {
        KexiDB::QueryColumnInfo* ci;
        const int indexForVisibleLookupValue = fields[i]->indexForVisibleLookupValue();
        if (-1 != indexForVisibleLookupValue) {
            ci = query->expandedOrInternalField(indexForVisibleLookupValue);
            visibleFieldIndex[i] = indexForVisibleLookupValue;
        } else {
            ci = fields[i];
            visibleFieldIndex[i] = i;
        }

        isText[i] = ci->field->isTextType();
        isDateTime[i] = ci->field->type() == KexiDB::Field::DateTime;
        isTime[i] = ci->field->type() == KexiDB::Field::Time;
        isBLOB[i] = ci->field->type() == KexiDB::Field::BLOB;
//  isInteger[i] = fields[i]->field->isIntegerType()
//   || fields[i]->field->type()==KexiDB::Field::Boolean;
//  isFloatingPoint[i] = fields[i]->field->isFPNumericType();
    }

kDebug() << 1;
    // 1. Output column names
    if (options.addColumnNames) {
        for (uint i = 0; i < fieldsCount; i++) {
            //kDebug() << "Adding column names";
            if (i > 0) {
                APPEND(delimiter);
            }

            if (hasTextQuote) {
                APPEND(textQuote + fields[i]->captionOrAliasOrName().replace(textQuote, escapedTextQuote) + textQuote);
            } else {
                APPEND(fields[i]->captionOrAliasOrName());
            }
        }
        APPEND_EOLN
    }

    KexiGUIMessageHandler handler;
    KexiDB::Cursor *cursor = conn->executeQuery(*query, queryParams);
    if (!cursor) {
        handler.showErrorMessage(conn);
        _ERR;
    }
    for (cursor->moveFirst(); !cursor->eof() && !cursor->error(); cursor->moveNext()) {
        //kDebug() << "Adding records";
        const uint realFieldCount = qMin(cursor->fieldCount(), fieldsCount);
        for (uint i = 0; i < realFieldCount; i++) {
            const uint real_i = visibleFieldIndex[i];
            if (i > 0) {
                APPEND(delimiter);
            }

            if (cursor->value(real_i).isNull()) {
                continue;
            }

            if (isText[real_i]) {
                if (hasTextQuote)
                    APPEND(textQuote + QString(cursor->value(real_i).toString()).replace(textQuote, escapedTextQuote) + textQuote);
                else
                    APPEND(cursor->value(real_i).toString());
            } else if (isDateTime[real_i]) { //avoid "T" in ISO DateTime
                APPEND(cursor->value(real_i).toDateTime().date().toString(Qt::ISODate) + " "
                       + cursor->value(real_i).toDateTime().time().toString(Qt::ISODate));
            } else if (isTime[real_i]) { //time is temporarily stored as null date + time...
                APPEND(cursor->value(real_i).toTime().toString(Qt::ISODate));
            } else if (isBLOB[real_i]) { //BLOB is escaped in a special way
                if (hasTextQuote)
//! @todo add options to suppport other types from KexiDB::BLOBEscapingType enum...
                    APPEND(textQuote + KexiDB::escapeBLOB(cursor->value(real_i).toByteArray(), KexiDB::BLOBEscapeHex) + textQuote);
                else
                    APPEND(KexiDB::escapeBLOB(cursor->value(real_i).toByteArray(), KexiDB::BLOBEscapeHex));
            } else {//other types
                APPEND(cursor->value(real_i).toString());
            }
        }
        APPEND_EOLN
    }

    if (copyToClipboard)
        buffer.squeeze();

    if (!conn->deleteCursor(cursor)) {
        handler.showErrorMessage(conn);
        _ERR;
    }

    if (copyToClipboard)
        kapp->clipboard()->setText(buffer, QClipboard::Clipboard);

    kDebug() << "Done";

    if (kSaveFile) {
        stream->flush();
        if (!kSaveFile->finalize()) {
            kWarning() << "Error finalizing stream!";
        }
    }
    return true;
}
