//# JsonParser.h: Class for parsing Json-style key:value lines
//# Copyright (C) 2016
//# Associated Universities, Inc. Washington DC, USA.
//#
//# This library is free software; you can redistribute it and/or modify it
//# under the terms of the GNU Library General Public License as published by
//# the Free Software Foundation; either version 2 of the License, or (at your
//# option) any later version.
//#
//# This library is distributed in the hope that it will be useful, but WITHOUT
//# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
//# License for more details.
//#
//# You should have received a copy of the GNU Library General Public License
//# along with this library; if not, write to the Free Software Foundation,
//# Inc., 675 Massachusetts Ave, Cambridge, MA 02139, USA.
//#
//# Correspondence concerning AIPS++ should be addressed as follows:
//#        Internet email: casa-feedback@nrao.edu.
//#        Postal address: AIPS++ Project Office
//#                        National Radio Astronomy Observatory
//#                        520 Edgemont Road
//#                        Charlottesville, VA 22903-2475 USA

#ifndef CASA_JSONPARSER_H
#define CASA_JSONPARSER_H

//# Includes
#include <casacore/casa/BasicSL/String.h>
#include <casacore/casa/Exceptions/Error.h>

extern "C" {
  int JsonGramwrap();        // yywrap
}

namespace casacore {

  //# Forward Declarations
  class JsonValue;
  class JSonKVMap;
  
  // <summary>
  // Class for parsing Json-style key:value lines.
  // </summary>

  // <use visibility=export>
  // <reviewed reviewer="" date="" tests="tJsonKVMap">
  // </reviewed>

  //# <prerequisite>
  //# </prerequisite>

  // <synopsis>
  // JsonParser is a class for parsing JSON files. Its function 'parse'
  // is the main function to do so.
  // It can handle any JSON file (not only those generated by JsonOut).
  // It supports (i.e., strips) possible comments in C, C++ and Python style.
  // It also supports complex numbers (structs with fields "r" and "i").
  // Escaped characters in a string value are translated into their ASCII counterparts.
  //
  // The result of the parser is a JsonKVMap object containing all fields
  // and values (scalars, arrays and structs, possibly nested in any way).
  // The values in the map are stored as JsonValue objects, which have functions to
  // get the value with the proper type.
  // </synopsis>

  // <example>
  // The following example is the opposite of the one given for class JsonOut.
  // <srcblock>
  // // Parse the given JSON file.
  // JsonKVMap jmap = JsonParser::parseFile (fileName);
  // // Check if the version is correct.
  // AlwaysAssert (jmap.getInt("Version", 1) == 1, AipsError);
  // uInt axis = jmap.get("Axis").getInt();
  // // Get the vector of names from the map and JsonValue.
  // Vector<String> names(jmap.get("Images").getArrayString());
  // </srcblock>
  // </example>

  // <motivation>
  // JSON is a commonly used interchange format.
  // However, commonly available parsers do not support data type Complex. Also, comments
  // are often not supported (alas standard Json does not allow comments).
  // </motivation>

  //# <todo asof="1996/03/10">
  //#   <li> 
  //# </todo>

  class JsonParser
  {
  public:
    // Parse the command in the given string and return the resulting map.
    static JsonKVMap parse (const String& command);
      
    // Parse the given file and return the resulting map.
    // Comments are ignored; they can be indicated by // or # till eol
    // or be enclosed in / * and * /.
    static JsonKVMap parseFile (const String& fileName);
      
    // Give the next chunk of input for the scanner.
    static int input (char* buf, int max_size);
      
    // Give the current position (for read or update).
    static int& position()
      { return theirPosition; }
      
    // Remove all possible escape characters and convert as needed (including <src>\uxxxx</src>).
    static String removeEscapes (const String& in);
      
    // Let the parser set the final KeyValueMap.
    static void setMap (JsonKVMap* map)
      { theirJsonMap = map; }
      
  private:
    static int theirPosition;
    static const char* theirCommand;
    static JsonKVMap* theirJsonMap;
  };
  
  // The global yyerror function for the parser.
  // It throws an exception with the current token.
  void JsonGramerror (const char*);
  
  // </group>

} // end namespace

#endif
