/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/

//BL_COPYRIGHT_NOTICE
#include <winstd.H>

#include <new>
#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <list>
#include <map>
#include <algorithm>
#include <fstream>

#ifndef WIN32
#include <unistd.h>
#endif

#include "ParmParse.H"
#include "ParallelDescriptor.H"
#include "DataServices.H"
#include "Utility.H"
#include "FArrayBox.H"
#include "Utility.H"
#include "ArrayLim.H"

#if defined(BL_FORT_USE_UPPERCASE)
#  define WRITEVEL  WRITEVEL
#elif defined(BL_FORT_USE_LOWERCASE)
#  define WRITEVEL  writevel
#elif defined(BL_FORT_USE_UNDERSCORE)
#  define WRITEVEL  writevel_
#endif

extern "C" {
    void WRITEVEL(const Real* dat, ARLIM_P(lo),ARLIM_P(hi),
                  const int* lo, const int* hi, const Real* probsize);
};

static
void 
print_usage (int,
             char* argv[])
{
	std::cerr << "usage:\n";
    std::cerr << argv[0] << " infile=<filename>\n";
    exit(1);
}

void
main (int   argc,
      char* argv[])
{
    BoxLib::Initialize(argc,argv);

    if (argc < 2)
        print_usage(argc,argv);

    ParmParse pp;

    if (pp.contains("help"))
        print_usage(argc,argv);

    if (pp.contains("verbose"))
        AmrData::SetVerbose(true);

    std::string infile;
    pp.get("infile",infile);

    std::cout << "Reading " << infile << "...";

    DataServices::SetBatchMode();
    FileType fileType(NEWPLT);
    DataServices dataServices(infile, fileType);
    std::cout << "done" << std::endl;

    if (!dataServices.AmrDataOk())
        DataServices::Dispatch(DataServices::ExitRequest, NULL);
    AmrData& amrData = dataServices.AmrDataRef();

    int finestLevel = amrData.FinestLevel();
    Box box(amrData.ProbDomain()[finestLevel]);
    Array<std::string> plotnames = amrData.PlotVarNames();	
    FArrayBox vFab(box,BL_SPACEDIM);
    BoxArray ba(1);
    ba.set(0,box);
    MultiFab TMP(ba,1,0);
    
    const Array<Real>& dx = amrData.DxLevel()[finestLevel];
    
    std::cout << "Loading v into fab...";
    int idV = -1;
    for (int i=0; i<plotnames.size(); ++i)
        if (plotnames[i] == "x_velocity") idV = i;
    
    std::string dirName[3] = {"x", "y", "z"};
    for (int d=0; d<BL_SPACEDIM; ++d)
    {
        std::string name = dirName[d]+std::string("_velocity");
        amrData.FillVar(TMP, amrData.FinestLevel(), name);
        TMP.copy(vFab,0,d,1);
    }
    std::cout << "done" << std::endl;


    if (ParallelDescriptor::IOProcessor())
    {
        cout << "...writing turbulence file..." << endl;
        
        WRITEVEL(vFab.dataPtr(),ARLIM(vFab.loVect()),ARLIM(vFab.hiVect()),
                 box.loVect(), box.hiVect(), amrData.ProbSize().dataPtr());
        
        cout << "done." << endl;
    }

    BoxLib::Finalize();
}
