// SPDX-License-Identifier: GPL-3.0-or-later
// Copyright 2021 Juan Palacios <jpalaciosdev@gmail.com>

#include "pmvoltcurveprovider.h"

#include "../pmoverdriveprovider.h"
#include "common/fileutils.h"
#include "core/components/amdutils.h"
#include "core/info/amd/gpuinfopmoverdrive.h"
#include "core/info/igpuinfo.h"
#include "core/sysfsdatasource.h"
#include "pmvoltcurve.h"
#include <easylogging++.h>
#include <filesystem>
#include <fmt/format.h>
#include <memory>
#include <string>
#include <vector>

std::vector<std::unique_ptr<IControl>>
AMD::PMVoltCurveProvider::provideGPUControls(IGPUInfo const &gpuInfo,
                                             ISWInfo const &) const
{
  std::vector<std::unique_ptr<IControl>> controls;

  if (gpuInfo.vendor() == Vendor::AMD &&
      gpuInfo.hasCapability(GPUInfoPMOverdrive::VoltCurve)) {

    auto ppOdClkVolt = gpuInfo.path().sys / "pp_od_clk_voltage";
    auto ppOdClkVoltLines = Utils::File::readFileLines(ppOdClkVolt);

    auto voltCurveControlValid =
        !Utils::AMD::ppOdClkVoltageHasKnownVoltCurveQuirks(ppOdClkVoltLines) &&
        Utils::AMD::parseOverdriveVoltCurveRange(ppOdClkVoltLines).has_value() &&
        Utils::AMD::parseOverdriveVoltCurve(ppOdClkVoltLines).has_value();

    if (voltCurveControlValid) {

      controls.emplace_back(std::make_unique<AMD::PMVoltCurve>(
          "vc", std::make_unique<SysFSDataSource<std::vector<std::string>>>(
                    ppOdClkVolt)));
    }
    else {
      LOG(WARNING) << fmt::format("Invalid data on {}", ppOdClkVolt.string());
      for (auto const &line : ppOdClkVoltLines)
        LOG(ERROR) << line;
    }
  }

  return controls;
}

bool const AMD::PMVoltCurveProvider::registered_ =
    AMD::PMOverdriveProvider::registerProvider(
        std::make_unique<AMD::PMVoltCurveProvider>());
