"""Create a constant-valued function with given value."""

# Copyright (C) 2008-2009 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Johan Hake, 2008.
#
# First added:  2008-12-02
# Last changed: 2010-08-16

__all__ = ["Constant"]

# Import UFL and SWIG-generated extension module (DOLFIN C++)
import ufl
import ufl.domains
import dolfin.cpp as cpp
import numpy

class Constant(ufl.Coefficient, cpp.Constant):

    # TODO: [martinal] I want to change cell to domain, but that will break user code...
    def __init__(self, value, cell=None, name=None):
        """
        Create constant-valued function with given value.

        *Arguments*
            value
                The value may be either a single scalar value, or a
                tuple/list of values for vector-valued functions, or
                nested lists or a numpy array for tensor-valued
                functions.
            cell
                Optional argument. A :py:class:`Cell
                <ufl.Cell>` which defines the geometrical
                dimensions the Constant is defined for.

            name
                Optional argument. A str which overrules the default
                name of the Constant.
                
        The data type Constant represents a constant value that is
        unknown at compile-time. Its values can thus be changed
        without requiring re-generation and re-compilation of C++
        code.

        *Examples of usage*

            .. code-block:: python

                p = Constant(pi/4)              # scalar
                C = Constant((0.0, -1.0, 0.0))  # constant vector

        """

        # Check that the type of the cell is ufl.Cell if given
        if cell is not None:
            if not isinstance(cell, (ufl.Cell, ufl.domains.DomainDescription)):
                raise TypeError("Expected an ufl.Cell as the second argument")

        array = numpy.array(value)
        dim = len(array.shape)
        floats = map(float, array.flat)

        # Create UFL element and initialize constant
        if dim == 0:
            self._ufl_element = ufl.FiniteElement("Real", \
                                                  cell, 0)
            cpp.Constant.__init__(self, floats[0])
        elif dim == 1:
            self._ufl_element = ufl.VectorElement("Real", \
                                                  cell, 0, len(floats))
            cpp.Constant.__init__(self, floats)
        else:
            self._ufl_element = ufl.TensorElement("Real", \
                                                  cell, 0, shape=array.shape)
            cpp.Constant.__init__(self, list(array.shape), floats)

        # Initialize base classes
        ufl.Coefficient.__init__(self, self._ufl_element)

        # Set name as given or automatic
        name = name or "f_%d" % self.count()
        self.rename(name, "a Constant")

    def __float__(self):
        # Overriding UFL operator in this particular case.
        if self.shape() != ():
            raise TypeError("Cannot convert nonscalar constant to float.")
        return cpp.Constant.__float__(self)

    def __str__(self):
        "x.__str__() <==> print(x)"
        return self.name()
        
