/* $Id: CheckLoops.cpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Check, if next statements occur within a loop statement.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "frontend/visitor/CheckLoops.hpp"
#include "frontend/ast/LoopStat.hpp"
#include "frontend/ast/NextStat.hpp"
#include "frontend/ast/ExitStat.hpp"
#include "frontend/reporting/ErrorRegistry.hpp"

namespace ast {

template <typename T>
void
CheckLoops::visitLoopCFNode(T &node, const char *kind) const
{
	LoopStat *referred = this->lookup(node.loopLabel);
	if (referred == NULL) {
		this->missingLoop(node, node.loopLabel, kind);
		return;
	}

	assert(node.referredLoop == NULL);
	node.referredLoop = referred;
}

void
CheckLoops::visit(ExitStat &node)
{
	this->visitLoopCFNode(node, "Exit");
}

void
CheckLoops::visit(NextStat &node)
{
	this->visitLoopCFNode(node, "Next");
}

void
CheckLoops::process(LoopStat &node)
{
	this->loops.push_front(&node);
	assert(node.loopStats != NULL);
	if (node.loopStats->empty()) {
		// issue warning?
	}

	this->listTraverse(*node.loopStats);
	this->loops.pop_front();
}

LoopStat *
CheckLoops::lookup(SimpleName *label) const
{
	if (label == NULL) {
		if (this->loops.empty()) {
			return NULL;
		}
		return this->loops.front();
	}

	for (std::list<LoopStat*>::const_iterator i = this->loops.begin();
		i != this->loops.end(); i++) {
		
		if (((*i)->name != NULL) && (*(*i)->name == *label->name)) {
			return *i;
		}
	}
	return NULL;
}

void
CheckLoops::missingLoop(
	const AstNode &node, 
	const SimpleName *label,
	const char *kind
) const
{
	std::string msg;

	if (label == NULL) {
		msg = kind;
		msg += " statement not within a loop statement.";
	} else {
		assert(label->name != NULL);
		msg = "Missing loop statement with label <" + *label->name 
			+ ">.";
	}
	CompileError *ce = new CompileError(node, msg);
	ErrorRegistry::addError(ce);
}

}; /* namespace ast */
