/*
 * File    : ProjectSettingsEditorPanel.java
 * Created : 27-feb-2004 10:33
 * By      : fbusquets
 *
 * JClic - Authoring and playing system for educational activities
 *
 * Copyright (C) 2000 - 2018 Francesc Busquets & Departament
 * d'Educacio de la Generalitat de Catalunya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details (see the LICENSE file).
 */
package edu.xtec.jclic.project;

import edu.xtec.jclic.bags.MediaBagEditor;
import edu.xtec.jclic.beans.*;
import edu.xtec.jclic.edit.Editor;
import edu.xtec.jclic.edit.EditorPanel;
import edu.xtec.jclic.media.EventSounds;
import edu.xtec.util.Messages;
import edu.xtec.util.Options;
import edu.xtec.util.StrUtils;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;

/**
 * @author Francesc Busquets (fbusquets@xtec.cat)
 * @version 13.10.29
 */
public class ProjectSettingsEditorPanel extends EditorPanel {

  private javax.swing.JTextPane[] descPanels;
  private Locale[] meta_langs = new Locale[] {};
  private javax.swing.JCheckBox[] levelChkBoxes;
  private javax.swing.JCheckBox[] areaChkBoxes;

  /** Creates new form JClicProjectEditorPanel */
  public ProjectSettingsEditorPanel(Options options) {
    super(options);
    initComponents();
    // Order based in ProjectSettings.KNOWN_LEVEL_CODES
    levelChkBoxes = new javax.swing.JCheckBox[] { levels_INF, levels_PRI, levels_SEC, levels_BTX };
    areaChkBoxes = new javax.swing.JCheckBox[] { areas_lleng, areas_mat, areas_soc, areas_exp, areas_mus, areas_vip,
        areas_ef, areas_tec, areas_div };
    listenButtonActions(levelChkBoxes);
    listenButtonActions(areaChkBoxes);
    evSoundsBtn.setOptions(options);
    postInit(250, false, false);
    setEnabled(false);
  }

  /**
   * This method is called from within the constructor to initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is always
   * regenerated by the Form Editor.
   */
  // <editor-fold defaultstate="collapsed" desc="Generated
  // Code">//GEN-BEGIN:initComponents
  private void initComponents() {
    java.awt.GridBagConstraints gridBagConstraints;

    scroll = new javax.swing.JScrollPane();
    scroll.getVerticalScrollBar().setUnitIncrement(16);
    mainPanel = new javax.swing.JPanel();
    descrPanel = new edu.xtec.jclic.beans.RollPanel();
    javax.swing.JLabel titleLb = new javax.swing.JLabel();
    titleText = new javax.swing.JTextField();
    javax.swing.JLabel descLb = new javax.swing.JLabel();
    tabDesc = new javax.swing.JTabbedPane();
    addBtn = new javax.swing.JButton();
    removeBtn = new javax.swing.JButton();
    coverPanel = new edu.xtec.jclic.beans.RollPanel();
    imgPanel1 = new javax.swing.JPanel();
    javax.swing.JLabel coverLb = new javax.swing.JLabel();
    coverButton = new edu.xtec.jclic.beans.ImgButton();
    coverButton.setOptions(options);
    javax.swing.JLabel thumbLb = new javax.swing.JLabel();
    thumbButton = new edu.xtec.jclic.beans.ImgButton();
    thumbButton.setOptions(options);
    imgPanel2 = new javax.swing.JPanel();
    javax.swing.JLabel icon16Lb = new javax.swing.JLabel();
    icon16Button = new edu.xtec.jclic.beans.ImgButton();
    icon16Button.setOptions(options);
    javax.swing.JLabel icon72Lb = new javax.swing.JLabel();
    icon72Button = new edu.xtec.jclic.beans.ImgButton();
    icon72Button.setOptions(options);
    javax.swing.JLabel icon192Lb = new javax.swing.JLabel();
    icon192Button = new edu.xtec.jclic.beans.ImgButton();
    icon192Button.setOptions(options);
    authPanel = new edu.xtec.jclic.beans.RollPanel();
    javax.swing.JLabel authorLb = new javax.swing.JLabel();
    authorListEditor = new TextListEditor(options) {
      protected Object editItem(Object currentValue, boolean newValue) {
        return editAuthor(currentValue, newValue);
      }
    };
    javax.swing.JLabel orgLb = new javax.swing.JLabel();
    orgListEditor = new TextListEditor(options) {
      protected Object editItem(Object currentValue, boolean newValue) {
        return editOrg(currentValue, newValue);
      }
    };
    javax.swing.JLabel revLb = new javax.swing.JLabel();
    revListEditor = new TextListEditor(options) {
      protected Object editItem(Object currentValue, boolean newValue) {
        return editRevision(currentValue, newValue);
      }
    };
    javax.swing.JLabel licenseLb = new javax.swing.JLabel();
    licenseCombo = new javax.swing.JComboBox<String>(
        edu.xtec.jclic.project.ProjectSettings.getLicensesList(this.options.getMsg("settings_license_other")));
    licenseCombo.setSelectedIndex(edu.xtec.jclic.project.ProjectSettings.CC_BY_NC_SA);
    descPanel = new edu.xtec.jclic.beans.RollPanel();
    javax.swing.JLabel langLb = new javax.swing.JLabel();
    langListEditor = new TextListEditor(options) {
      protected Object editItem(Object currentValue, boolean newValue) {
        return editLanguage(currentValue, newValue);
      }
    };
    javax.swing.JLabel levelLb = new javax.swing.JLabel();
    levels_INF = new javax.swing.JCheckBox();
    levels_PRI = new javax.swing.JCheckBox();
    levels_SEC = new javax.swing.JCheckBox();
    levels_BTX = new javax.swing.JCheckBox();
    javax.swing.JLabel levelOtherLb = new javax.swing.JLabel();
    levelText = new javax.swing.JTextField();
    javax.swing.JLabel areaLb = new javax.swing.JLabel();
    areas_lleng = new javax.swing.JCheckBox();
    areas_mat = new javax.swing.JCheckBox();
    areas_soc = new javax.swing.JCheckBox();
    areas_exp = new javax.swing.JCheckBox();
    areas_mus = new javax.swing.JCheckBox();
    areas_vip = new javax.swing.JCheckBox();
    areas_ef = new javax.swing.JCheckBox();
    areas_tec = new javax.swing.JCheckBox();
    areas_div = new javax.swing.JCheckBox();
    javax.swing.JLabel areaOtherLb = new javax.swing.JLabel();
    areaText = new javax.swing.JTextField();
    javax.swing.JLabel descriptorslLb = new javax.swing.JLabel();
    descriptorsText = new javax.swing.JTextField();
    uiPanel = new edu.xtec.jclic.beans.RollPanel();
    javax.swing.JLabel skinLb = new javax.swing.JLabel();
    skinCombo = new javax.swing.JComboBox<String>(edu.xtec.jclic.skins.Skin.getSystemSkinList(false));
    skinCombo.setSelectedItem(null);
    javax.swing.JLabel evSoundsLb = new javax.swing.JLabel();
    evSoundsBtn = new edu.xtec.jclic.beans.EventSoundsButton();
    javax.swing.JLabel spacer = new javax.swing.JLabel();

    setLayout(new java.awt.BorderLayout());

    mainPanel.setLayout(new java.awt.GridBagLayout());

    descrPanel.setTitle(options.getMsg("edit_project_description_block"));
    descrPanel.getMainPanel().setLayout(new java.awt.GridBagLayout());

    titleLb.setLabelFor(titleText);
    titleLb.setText(options.getMsg("edit_project_title"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    descrPanel.getMainPanel().add(titleLb, gridBagConstraints);

    titleText.setToolTipText(options.getMsg("edit_project_title_tooltip"));
    titleText.setPreferredSize(new java.awt.Dimension(500, 21));
    titleText.getDocument().addDocumentListener(this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    descrPanel.getMainPanel().add(titleText, gridBagConstraints);

    descLb.setText(options.getMsg("edit_project_description"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    descrPanel.getMainPanel().add(descLb, gridBagConstraints);

    tabDesc.setToolTipText(options.getMsg("edit_project_description_tooltip"));
    tabDesc.setPreferredSize(new java.awt.Dimension(500, 200));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    descrPanel.getMainPanel().add(tabDesc, gridBagConstraints);

    addBtn.setIcon(new javax.swing.ImageIcon(getClass().getResource("/edu/xtec/resources/icons/plus16.gif"))); // NOI18N
    addBtn.setToolTipText(options == null ? "" : options.getMsg("edit_list_newElement_tooltip"));
    addBtn.setMargin(new java.awt.Insets(0, 0, 0, 0));
    addBtn.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        addBtnActionPerformed(evt);
      }
    });
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 0);
    descrPanel.getMainPanel().add(addBtn, gridBagConstraints);

    removeBtn.setIcon(new javax.swing.ImageIcon(getClass().getResource("/edu/xtec/resources/icons/minus16.gif"))); // NOI18N
    removeBtn.setToolTipText(options == null ? "" : options.getMsg("edit_list_deleteElement_tooltip"));
    removeBtn.setEnabled(false);
    removeBtn.setMargin(new java.awt.Insets(0, 0, 0, 0));
    removeBtn.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        removeBtnActionPerformed(evt);
      }
    });
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descrPanel.getMainPanel().add(removeBtn, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    mainPanel.add(descrPanel, gridBagConstraints);

    coverPanel.setTitle(options.getMsg("edit_project_images_block"));
    coverPanel.getMainPanel().setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.LEADING, 0, 0));

    imgPanel1.setFocusable(false);
    imgPanel1.setLayout(new java.awt.GridBagLayout());

    coverLb.setLabelFor(coverButton);
    coverLb.setText(options.getMsg("edit_project_cover"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    imgPanel1.add(coverLb, gridBagConstraints);

    coverButton.setToolTipText(options.getMsg("edit_project_cover_tooltip"));
    coverButton.addPropertyChangeListener(ImgButton.PROP_IMG_NAME, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 30);
    imgPanel1.add(coverButton, gridBagConstraints);

    thumbLb.setLabelFor(thumbButton);
    thumbLb.setText(options.getMsg("edit_project_thumb"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    imgPanel1.add(thumbLb, gridBagConstraints);

    thumbButton.setToolTipText(options.getMsg("edit_project_thumb_tooltip"));
    thumbButton.addPropertyChangeListener(ImgButton.PROP_IMG_NAME, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 30, 3);
    imgPanel1.add(thumbButton, gridBagConstraints);

    coverPanel.getMainPanel().add(imgPanel1);

    imgPanel2.setLayout(new java.awt.GridBagLayout());

    icon16Lb.setLabelFor(icon16Button);
    icon16Lb.setText(options.getMsg("edit_project_icon16"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    imgPanel2.add(icon16Lb, gridBagConstraints);

    icon16Button.setToolTipText(options.getMsg("edit_project_icons_tooltip"));
    icon16Button.addPropertyChangeListener(ImgButton.PROP_IMG_NAME, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    imgPanel2.add(icon16Button, gridBagConstraints);

    icon72Lb.setLabelFor(icon72Button);
    icon72Lb.setText(options.getMsg("edit_project_icon72"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    imgPanel2.add(icon72Lb, gridBagConstraints);

    icon72Button.setToolTipText(options.getMsg("edit_project_icons_tooltip"));
    icon72Button.addPropertyChangeListener(ImgButton.PROP_IMG_NAME, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    imgPanel2.add(icon72Button, gridBagConstraints);

    icon192Lb.setLabelFor(icon192Button);
    icon192Lb.setText(options.getMsg("edit_project_icon192"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    imgPanel2.add(icon192Lb, gridBagConstraints);

    icon192Button.setToolTipText(options.getMsg("edit_project_icons_tooltip"));
    icon192Button.addPropertyChangeListener(ImgButton.PROP_IMG_NAME, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    imgPanel2.add(icon192Button, gridBagConstraints);

    coverPanel.getMainPanel().add(imgPanel2);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    mainPanel.add(coverPanel, gridBagConstraints);

    authPanel.setTitle(options.getMsg("edit_project_author_group"));
    authPanel.getMainPanel().setLayout(new java.awt.GridBagLayout());

    authorLb.setLabelFor(authorListEditor);
    authorLb.setText(options.getMsg("edit_project_authors"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    authPanel.getMainPanel().add(authorLb, gridBagConstraints);

    authorListEditor.setToolTipText(options.getMsg("edit_project_authors_tooltip"));
    authorListEditor.setPreferredSize(new java.awt.Dimension(500, 80));
    authorListEditor.addPropertyChangeListener(TextListEditor.PROP_LIST, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    gridBagConstraints.weightx = 1.0;
    authPanel.getMainPanel().add(authorListEditor, gridBagConstraints);

    orgLb.setLabelFor(orgListEditor);
    orgLb.setText(options.getMsg("edit_project_org"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    authPanel.getMainPanel().add(orgLb, gridBagConstraints);

    orgListEditor.setToolTipText(options.getMsg("edit_project_org_tooltip"));
    orgListEditor.setPreferredSize(new java.awt.Dimension(500, 80));
    orgListEditor.addPropertyChangeListener(TextListEditor.PROP_LIST, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    authPanel.getMainPanel().add(orgListEditor, gridBagConstraints);

    revLb.setLabelFor(revListEditor);
    revLb.setText(options.getMsg("edit_project_rev"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    authPanel.getMainPanel().add(revLb, gridBagConstraints);

    revListEditor.setToolTipText(options.getMsg("edit_project_rev_tooltip"));
    revListEditor.setPreferredSize(new java.awt.Dimension(500, 80));
    revListEditor.addPropertyChangeListener(TextListEditor.PROP_LIST, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    authPanel.getMainPanel().add(revListEditor, gridBagConstraints);

    licenseLb.setLabelFor(licenseCombo);
    licenseLb.setText(options.getMsg("settings_license"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    authPanel.getMainPanel().add(licenseLb, gridBagConstraints);

    licenseCombo.setToolTipText(options.getMsg("settings_license_tooltip"));
    licenseCombo.addActionListener(this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    authPanel.getMainPanel().add(licenseCombo, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    mainPanel.add(authPanel, gridBagConstraints);

    descPanel.setTitle(options.getMsg("edit_project_descriptors_block"));
    descPanel.getMainPanel().setLayout(new java.awt.GridBagLayout());

    langLb.setLabelFor(langListEditor);
    langLb.setText(options.getMsg("edit_project_languages"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    descPanel.getMainPanel().add(langLb, gridBagConstraints);

    langListEditor.setToolTipText(options.getMsg("edit_project_languages_tooltip"));
    langListEditor.setPreferredSize(new java.awt.Dimension(200, 80));
    langListEditor.addPropertyChangeListener(TextListEditor.PROP_LIST, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 0);
    descPanel.getMainPanel().add(langListEditor, gridBagConstraints);

    levelLb.setText(options.getMsg("edit_project_level"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    descPanel.getMainPanel().add(levelLb, gridBagConstraints);

    levels_INF.setText(options.getMsg("levels_INF"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(levels_INF, gridBagConstraints);

    levels_PRI.setText(options.getMsg("levels_PRI"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(levels_PRI, gridBagConstraints);

    levels_SEC.setText(options.getMsg("levels_SEC"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(levels_SEC, gridBagConstraints);

    levels_BTX.setText(options.getMsg("levels_BTX"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(levels_BTX, gridBagConstraints);

    levelOtherLb.setLabelFor(levelText);
    levelOtherLb.setText(options.getMsg("levels_other"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 10, 3);
    descPanel.getMainPanel().add(levelOtherLb, gridBagConstraints);

    levelText.setToolTipText(options.getMsg("edit_project_level_tooltip"));
    levelText.setPreferredSize(new java.awt.Dimension(300, 21));
    levelText.getDocument().addDocumentListener(this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 10, 3);
    descPanel.getMainPanel().add(levelText, gridBagConstraints);

    areaLb.setText(options.getMsg("edit_project_area"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    descPanel.getMainPanel().add(areaLb, gridBagConstraints);

    areas_lleng.setText(options.getMsg("areas_lleng"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_lleng, gridBagConstraints);

    areas_mat.setText(options.getMsg("areas_mat"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 2;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_mat, gridBagConstraints);

    areas_soc.setText(options.getMsg("areas_soc"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_soc, gridBagConstraints);

    areas_exp.setText(options.getMsg("areas_exp"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 2;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_exp, gridBagConstraints);

    areas_mus.setText(options.getMsg("areas_mus"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_mus, gridBagConstraints);

    areas_vip.setText(options.getMsg("areas_vip"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 2;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_vip, gridBagConstraints);

    areas_ef.setText(options.getMsg("areas_ef"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_ef, gridBagConstraints);

    areas_tec.setText(options.getMsg("areas_tec"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 2;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_tec, gridBagConstraints);

    areas_div.setText(options.getMsg("areas_div"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    descPanel.getMainPanel().add(areas_div, gridBagConstraints);

    areaOtherLb.setLabelFor(areaText);
    areaOtherLb.setText(options.getMsg("areas_other"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 10, 3);
    descPanel.getMainPanel().add(areaOtherLb, gridBagConstraints);

    areaText.setToolTipText(options.getMsg("edit_project_area_tooltip"));
    areaText.setPreferredSize(new java.awt.Dimension(300, 21));
    areaText.getDocument().addDocumentListener(this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 10, 3);
    descPanel.getMainPanel().add(areaText, gridBagConstraints);

    descriptorslLb.setLabelFor(descriptorsText);
    descriptorslLb.setText(options.getMsg("edit_project_descriptors"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 10, 3);
    descPanel.getMainPanel().add(descriptorslLb, gridBagConstraints);

    descriptorsText.setToolTipText(options.getMsg("edit_project_descriptors_tooltip"));
    descriptorsText.setPreferredSize(new java.awt.Dimension(500, 21));
    descriptorsText.getDocument().addDocumentListener(this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 10, 3);
    descPanel.getMainPanel().add(descriptorsText, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    mainPanel.add(descPanel, gridBagConstraints);

    uiPanel.setTitle(options.getMsg("edit_project_ui_block"));
    uiPanel.getMainPanel().setLayout(new java.awt.GridBagLayout());

    skinLb.setLabelFor(skinCombo);
    skinLb.setText(options.getMsg("settings_skin"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    uiPanel.getMainPanel().add(skinLb, gridBagConstraints);

    skinCombo.setEditable(true);
    skinCombo.setToolTipText(options.getMsg("settings_skin_tooltip"));
    skinCombo.addActionListener(this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    uiPanel.getMainPanel().add(skinCombo, gridBagConstraints);

    evSoundsLb.setLabelFor(revListEditor);
    evSoundsLb.setText(options.getMsg("edit_project_evsounds"));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    uiPanel.getMainPanel().add(evSoundsLb, gridBagConstraints);

    evSoundsBtn.setToolTipText(options.getMsg("edit_project_evsounds_tooltip"));
    evSoundsBtn.addPropertyChangeListener(EventSoundsButton.PROP_EVSND_NAME, this);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    uiPanel.getMainPanel().add(evSoundsBtn, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.weightx = 1.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    mainPanel.add(uiPanel, gridBagConstraints);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.gridheight = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.VERTICAL;
    gridBagConstraints.weighty = 1.0;
    mainPanel.add(spacer, gridBagConstraints);

    scroll.setViewportView(mainPanel);

    add(scroll, java.awt.BorderLayout.CENTER);
  } // </editor-fold>//GEN-END:initComponents

  private void addBtnActionPerformed(java.awt.event.ActionEvent evt) { // GEN-FIRST:event_addBtnActionPerformed

    String lang = (String) editLanguage(null, true);
    if (lang != null) {
      lang = Messages.getLanguageFromDescriptive(lang);
      Locale loc = Locale.forLanguageTag(lang);
      Locale[] ml = Arrays.copyOf(meta_langs, meta_langs.length + 1);
      for (Locale meta_lang : meta_langs) {
        if (meta_lang.equals(loc)) {
          options.getMessages().showAlert(this, "edit_project_languages_already_selected");
          return;
        }
      }
      ml[meta_langs.length] = loc;
      meta_langs = ml;
      addDescTab("");
    }
  } // GEN-LAST:event_addBtnActionPerformed

  private void removeBtnActionPerformed(java.awt.event.ActionEvent evt) { // GEN-FIRST:event_removeBtnActionPerformed

    if (meta_langs.length > 1) {
      int p = tabDesc.getSelectedIndex();
      if (this.options.getMessages().showQuestionDlg(this, "edit_project_languages_confirm_delete",
          meta_langs[p].getDisplayName(), "yn") == Messages.YES) {
        tabDesc.remove(p);
        Locale[] ml = new Locale[meta_langs.length - 1];
        for (int i = 0, c = 0; i < meta_langs.length; i++) {
          if (i != p) {
            ml[c++] = meta_langs[i];
          }
        }
        checkEnabled();
      }
    }
  } // GEN-LAST:event_removeBtnActionPerformed

  @Override
  public boolean checkIfEditorValid(Editor e) {
    return (e instanceof ProjectSettingsEditor);
  }

  protected ProjectSettingsEditor getProjectSettingsEditor() {
    return (ProjectSettingsEditor) getEditor();
  }

  protected ProjectSettings getProjectSettings() {
    ProjectSettings result = null;
    ProjectSettingsEditor pse = getProjectSettingsEditor();
    if (pse != null) {
      result = pse.getProjectSettings();
    }
    return result;
  }

  @Override
  protected void fillData() {
    ProjectSettings ps = getProjectSettings();
    JClicProjectEditor prjed = (ps == null ? null : getProjectSettingsEditor().getProjectEditor());
    MediaBagEditor mbe = (prjed == null ? null : prjed.getMediaBagEditor());

    if (prjed != null) {
      evSoundsBtn.setMediaBagEditor(mbe);
    }
    evSoundsBtn.setEventSounds(ps == null ? null : ps.eventSounds);

    coverButton.setMediaBagEditor(mbe);
    coverButton.setImgName(ps == null ? null : ps.coverFileName);

    thumbButton.setMediaBagEditor(mbe);
    thumbButton.setImgName(ps == null ? null : ps.thumbnailFileName);

    icon16Button.setMediaBagEditor(mbe);
    icon16Button.setImgName(ps == null ? null : ps.icon16);

    icon72Button.setMediaBagEditor(mbe);
    icon72Button.setImgName(ps == null ? null : ps.icon72);

    icon192Button.setMediaBagEditor(mbe);
    icon192Button.setImgName(ps == null ? null : ps.icon192);

    titleText.setText(ps == null || ps.title == null ? options.getMsg("UNNAMED") : ps.title);

    tabDesc.removeAll();
    descPanels = null;
    meta_langs = new Locale[] {};

    if (ps != null) {
      meta_langs = new Locale[ps.meta_langs.length];
      descPanels = new javax.swing.JTextPane[ps.meta_langs.length];
      for (int i = 0; i < meta_langs.length; i++) {
        meta_langs[i] = ps.meta_langs[i];
        addDescTab(ps.descriptions[i]);
      }
    }

    for (int i = 0; i < levelChkBoxes.length; i++)
      levelChkBoxes[i].setSelected(
          ps != null && ps.level_codes != null && ps.level_codes.contains(ProjectSettings.KNOWN_LEVEL_CODES[i]));

    for (int i = 0; i < areaChkBoxes.length; i++)
      areaChkBoxes[i].setSelected(
          ps != null && ps.area_codes != null && ps.area_codes.contains(ProjectSettings.KNOWN_AREA_CODES[i]));

    areaText.setText(ps == null ? "" : StrUtils.secureString(ps.area));
    levelText.setText(ps == null ? "" : StrUtils.secureString(ps.level));
    descriptorsText.setText(ps == null ? "" : StrUtils.secureString(ps.descriptors));
    skinCombo.setSelectedItem(ps == null ? null : ps.skinFileName);
    licenseCombo.setSelectedIndex(ps == null ? edu.xtec.jclic.project.ProjectSettings.CC_BY_NC_SA : ps.license);

    List<Object> v = new ArrayList<Object>();
    if (ps != null && ps.languages != null) {
      for (String language : ps.languages) {
        String code = (String) Messages.getNamesToCodes().get(language.toLowerCase());
        if (code != null) {
          v.add(Messages.getDescriptiveLanguageCode(code));
        } else {
          v.add(language);
        }
      }
    }
    langListEditor.setListData(v);

    v.clear();
    if (ps != null && ps.authors != null) {
      v.addAll(Arrays.asList(ps.authors));
    }

    authorListEditor.setListData(v);

    v.clear();
    if (ps != null && ps.organizations != null) {
      v.addAll(Arrays.asList(ps.organizations));
    }

    orgListEditor.setListData(v);

    v.clear();
    if (ps != null && ps.revisions != null) {
      v.addAll(Arrays.asList(ps.revisions));
    }

    revListEditor.setListData(v);
  }

  private void addDescTab(String text) {
    int i = tabDesc.getTabCount();

    if (i >= descPanels.length) {
      javax.swing.JTextPane[] dp = new javax.swing.JTextPane[descPanels.length + 1];
      for (int p = 0; p < descPanels.length; p++) {
        dp[p] = descPanels[p];
      }
      descPanels = dp;
    }

    javax.swing.JScrollPane scr = new javax.swing.JScrollPane();
    descPanels[i] = new javax.swing.JTextPane();
    descPanels[i].setText(StrUtils.secureString(text));
    descPanels[i].getDocument().addDocumentListener(this);
    scr.setViewportView(descPanels[i]);
    scr.setHorizontalScrollBarPolicy(javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
    tabDesc.addTab(meta_langs[i].toLanguageTag(), scr);
    checkEnabled();
  }

  @Override
  public void setEnabled(boolean enabled) {
    super.setEnabled(enabled);
    evSoundsBtn.setEnabled(enabled);
    titleText.setEnabled(enabled);
    tabDesc.setEnabled(enabled);
    addBtn.setEnabled(enabled);
    removeBtn.setEnabled(enabled && meta_langs != null && meta_langs.length > 1);
    descriptorsText.setEnabled(enabled);
    skinCombo.setEnabled(enabled);
    licenseCombo.setEnabled(enabled);
    langListEditor.setEnabled(enabled);
    authorListEditor.setEnabled(enabled);
    orgListEditor.setEnabled(enabled);
    revListEditor.setEnabled(enabled);
    coverButton.setEnabled(enabled);
    thumbButton.setEnabled(enabled);
    icon16Button.setEnabled(enabled);
    icon72Button.setEnabled(enabled);
    icon192Button.setEnabled(enabled);

    for (javax.swing.JCheckBox check : levelChkBoxes) {
      check.setEnabled(enabled);
    }
    levelText.setEnabled(enabled);

    for (javax.swing.JCheckBox check : areaChkBoxes) {
      check.setEnabled(enabled);
    }
    areaText.setEnabled(enabled);
  }

  protected void checkEnabled() {
    removeBtn.setEnabled(meta_langs != null && meta_langs.length > 1);
  }

  /*
   * public boolean isModified(){ boolean result=super.isModified();
   * ProjectSettings ps=getProjectSettings(); if(ps!=null && !result){ result=
   * !titleText.getText().equals(ps.title) ||
   * !descText.getText().equals(ps.description) ||
   * !areaText.getText().equals(ps.area) || !levelText.getText().equals(ps.level)
   * || !descriptorsText.getText().equals(ps.descriptors) ||
   * !Utils.compareObjects(skinCombo.getSelectedItem(), ps.skinFileName) ||
   * langListEditor.isModified() || authorListEditor.isModified() ||
   * orgListEditor.isModified() || revListEditor.isModified() ||
   * evSoundsBtn.isModified(); } return result; }
   */
  @Override
  protected void saveData() {
    ProjectSettings ps = getProjectSettings();
    if (ps != null) {
      ps.title = StrUtils.secureString(titleText.getText(), options.getMsg("UNNAMED"));

      ps.meta_langs = meta_langs;
      ps.descriptions = new String[meta_langs.length];
      for (int i = 0; i < meta_langs.length; i++) {
        ps.descriptions[i] = descPanels[i].getText();
      }
      ps.description = ps.descriptions[0];

      ps.coverFileName = coverButton.getImgName();
      ps.thumbnailFileName = thumbButton.getImgName();

      ps.icon16 = icon16Button.getImgName();
      ps.icon72 = icon72Button.getImgName();
      ps.icon192 = icon192Button.getImgName();

      List<Object> v = authorListEditor.getListData();
      ps.authors = (Author[]) v.toArray(new Author[v.size()]);
      v = orgListEditor.getListData();
      ps.organizations = (Organization[]) v.toArray(new Organization[v.size()]);
      v = revListEditor.getListData();
      ps.revisions = (Revision[]) v.toArray(new Revision[v.size()]);

      ps.level_codes = new ArrayList<String>();
      for (int i = 0; i < levelChkBoxes.length; i++)
        if (levelChkBoxes[i].isSelected())
          ps.level_codes.add(ProjectSettings.KNOWN_LEVEL_CODES[i]);
      ps.level = StrUtils.nullableString(levelText.getText());

      ps.area_codes = new ArrayList<String>();
      for (int i = 0; i < areaChkBoxes.length; i++)
        if (areaChkBoxes[i].isSelected())
          ps.area_codes.add(ProjectSettings.KNOWN_AREA_CODES[i]);
      ps.area = StrUtils.nullableString(areaText.getText());

      ps.descriptors = StrUtils.nullableString(descriptorsText.getText());
      String[] langs = langListEditor.getTextList();
      ps.languages = new String[langs.length];
      for (int i = 0; i < langs.length; i++) {
        String ln = langs[i];
        int p = ln.length();
        if (p > 5) {
          String code = ln.substring(p - 3, p - 1);
          ln = (String) Messages.getNamesToCodes().get(code);
        }
        ps.languages[i] = (ln == null ? langs[i] : ln);
      }
      ps.skinFileName = StrUtils.nullableString(skinCombo.getSelectedItem());
      ps.license = licenseCombo.getSelectedIndex();

      EventSounds evs = evSoundsBtn.getEventSounds();
      ps.eventSounds = (evs == null ? new EventSounds(null) : evs);
    }
  }

  private static JScrollPane langListPanel;
  private static JList<String> langList;

  private Object editLanguage(final Object currentValue, boolean newValue) {
    if (langList == null || langListPanel == null) {
      langList = new JList<String>(options.getMessages().getDescriptiveLanguageCodes());
      langListPanel = new JScrollPane(langList);
    }
    Object result = null;
    if (!newValue && currentValue != null) {
      SwingUtilities.invokeLater(new Runnable() {
        @Override
        public void run() {
          langList.setSelectedValue(currentValue, true);
        }
      });
    } else {
      langList.setSelectedIndex(-1);
    }
    if (options.getMessages().showInputDlg(this, new String[] { "edit_project_languages_select" }, null,
        new JComponent[] { langListPanel },
        (newValue ? "edit_project_languages_add" : "edit_project_languages_modify"))) {
      result = langList.getSelectedValue();
    }
    return result;
  }

  private AuthorEditPanel authorEditPanel;

  private Object editAuthor(Object currentValue, boolean newValue) {
    Author result = null;
    if (authorEditPanel == null) {
      authorEditPanel = new AuthorEditPanel(options);
    }
    if (newValue) {
      currentValue = null;
    }
    authorEditPanel.setAuthor((Author) currentValue);
    if (options.getMessages().showInputDlg(this, new String[] { "edit_project_author_info" }, null,
        new JComponent[] { authorEditPanel }, (newValue ? "edit_project_author_add" : "edit_project_author_edit"))) {
      result = authorEditPanel.getAuthor();
      if (result.name.length() < 1) {
        result = null;
      }
    }
    return result;
  }

  private OrganizationEditPanel organizationEditPanel;

  private Object editOrg(Object currentValue, boolean newValue) {
    Organization result = null;
    if (organizationEditPanel == null) {
      organizationEditPanel = new OrganizationEditPanel(options);
    }
    if (newValue) {
      currentValue = null;
    }
    organizationEditPanel.setOrganization((Organization) currentValue);
    if (options.getMessages().showInputDlg(this, new String[] { "edit_project_org_info" }, null,
        new JComponent[] { organizationEditPanel }, (newValue ? "edit_project_org_add" : "edit_project_org_edit"))) {
      result = organizationEditPanel.getOrganization();
      if (result.name.length() < 1) {
        result = null;
      }
    }
    return result;
  }

  private RevisionEditPanel revisionEditPanel;

  private Object editRevision(Object currentValue, boolean newValue) {
    Revision result = null;
    if (revisionEditPanel == null) {
      revisionEditPanel = new RevisionEditPanel(options);
    }
    if (newValue) {
      currentValue = null;
    }
    revisionEditPanel.setRevision((Revision) currentValue);
    if (options.getMessages().showInputDlg(this, new String[] { "edit_project_rev_info" }, null,
        new JComponent[] { revisionEditPanel }, (newValue ? "edit_project_rev_add" : "edit_project_rev_edit"))) {
      result = revisionEditPanel.getRevision();
      if (result != null && result.description.length() < 1) {
        result = null;
      }
    }
    return result;
  }

  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JButton addBtn;
  private javax.swing.JTextField areaText;
  private javax.swing.JCheckBox areas_div;
  private javax.swing.JCheckBox areas_ef;
  private javax.swing.JCheckBox areas_exp;
  private javax.swing.JCheckBox areas_lleng;
  private javax.swing.JCheckBox areas_mat;
  private javax.swing.JCheckBox areas_mus;
  private javax.swing.JCheckBox areas_soc;
  private javax.swing.JCheckBox areas_tec;
  private javax.swing.JCheckBox areas_vip;
  private edu.xtec.jclic.beans.RollPanel authPanel;
  private edu.xtec.jclic.beans.TextListEditor authorListEditor;
  private edu.xtec.jclic.beans.ImgButton coverButton;
  private edu.xtec.jclic.beans.RollPanel coverPanel;
  private edu.xtec.jclic.beans.RollPanel descPanel;
  private edu.xtec.jclic.beans.RollPanel descrPanel;
  private javax.swing.JTextField descriptorsText;
  private edu.xtec.jclic.beans.EventSoundsButton evSoundsBtn;
  private edu.xtec.jclic.beans.ImgButton icon16Button;
  private edu.xtec.jclic.beans.ImgButton icon192Button;
  private edu.xtec.jclic.beans.ImgButton icon72Button;
  private javax.swing.JPanel imgPanel1;
  private javax.swing.JPanel imgPanel2;
  private edu.xtec.jclic.beans.TextListEditor langListEditor;
  private javax.swing.JTextField levelText;
  private javax.swing.JCheckBox levels_BTX;
  private javax.swing.JCheckBox levels_INF;
  private javax.swing.JCheckBox levels_PRI;
  private javax.swing.JCheckBox levels_SEC;
  private javax.swing.JComboBox licenseCombo;
  private javax.swing.JPanel mainPanel;
  private edu.xtec.jclic.beans.TextListEditor orgListEditor;
  private javax.swing.JButton removeBtn;
  private edu.xtec.jclic.beans.TextListEditor revListEditor;
  private javax.swing.JScrollPane scroll;
  private javax.swing.JComboBox skinCombo;
  private javax.swing.JTabbedPane tabDesc;
  private edu.xtec.jclic.beans.ImgButton thumbButton;
  private javax.swing.JTextField titleText;
  private edu.xtec.jclic.beans.RollPanel uiPanel;
  // End of variables declaration//GEN-END:variables

}
