/***************************************************************************
                          chatmessage.h -  description
                             -------------------
    begin                : Sat Okt 29 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : "vdboor" --at-- "codingdomain.com"
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CHATMESSAGE_H
#define CHATMESSAGE_H

#include <QDateTime>
#include <QFont>
#include <QString>


class ContactBase;



/**
 * A class describing a chat message.
 *
 * @author Diederik van der Boor
 * @ingroup NetworkCore
 */
class ChatMessage
{
  public:
    // Defines the main categories into which messages are divided.
    enum MessageType
    {
      TYPE_INCOMING,           // Incoming IM message
      TYPE_OUTGOING,           // Outgoing IM message
      TYPE_APPLICATION,        // Generic application's message
      TYPE_APPLICATION_FILE,   // File transfer message
      TYPE_NOTIFICATION,       // Message events notification: winks, nudges, ...
      TYPE_PRESENCE,           // Presence events: contact join and leave events, ...
      TYPE_SYSTEM,             // Protocol messages: user is offline, message sending failed, ...
      TYPE_OFFLINE_INCOMING    // Incoming offline IM Message
    };

    // Define the specific kinds of contents that the message can hold
    enum ContentsClass
    {
      CONTENT_MESSAGE,
      CONTENT_MESSAGE_INK,                    // Ink / handwriting

      CONTENT_NOTIFICATION_NUDGE,             // Nudge
      CONTENT_NOTIFICATION_WINK,              // Wink
      CONTENT_PRESENCE_JOIN,                  // A contact has joined the chat
      CONTENT_PRESENCE_LEAVE,                 // A contact has left the chat
      CONTENT_PRESENCE_STATUS,                // A contact has changed status

      CONTENT_SYSTEM_NOTICE,                  // KMess or protocol level message about a contact, like "contact is blocking you" or "contact is offline"
      CONTENT_SYSTEM_ERROR,                   // Protocol/application error message, like "unsupported feature" or "message cannot be sent"

      // Generic application messages
      CONTENT_APP_INFO,               // Informative message (Connecting to host:port)
      CONTENT_APP_INVITE,             // An invitation for an application
      CONTENT_APP_STARTED,            // The invitation was accepted, the application will start
      CONTENT_APP_ENDED,              // The application ended successfully
      CONTENT_APP_FAILED,             // Some error made the application to fail
      CONTENT_APP_CANCELED            // The application or invitation has been canceled
    };

    // The constructor
                        ChatMessage( const MessageType      type,
                                     const ContentsClass    contentsClass,
                                           bool             isIncoming,
                                     const QString &        body,
                                     const QString &        contactHandle,
                                     const QString &        contactName = QString::null,
                                     const QString &        contactPicturePath = QString::null,
                                     const QFont &          font = QFont(),
                                     const QString &        fontColor = QString::null,
                                     const QDateTime &      time = QDateTime::currentDateTime() );

    // The destructor
                       ~ChatMessage();
    // A clone method for ChatView::showMessage()
    ChatMessage *       clone() const;

    // Return the message body
    const QString &     getBody() const;
    // Return the message date and time (for offline-im messages)
    const QDateTime &   getDateTime() const;
    // Return the handle of the sending contact
    const QString &     getContactHandle() const;
    // Return the name of the sending contact
    const QString &     getContactName() const;
    // Return the path of the sending contact's picture
    const QString &     getContactPicturePath() const;
    // Return the message contents class
    ContentsClass       getContentsClass() const;
    // Return the message font
    const QFont &       getFont() const;
    // Return the message font color
    const QString &     getFontColor() const;
    // Return the message time
    const QTime         getTime() const;
    // Return the message type
    MessageType         getType() const;
    // Return whether the body is already in HTML format.
    bool                hasHtmlBody() const;
    // Return whether the message is incoming or outgoing
    bool                isIncoming() const;
    // Return whether it's a normal chat message
    bool                isNormalMessage() const;


  private:
    // The message body
    QString             body_;
    // The handle of the sending contact
    QString             contactHandle_;
    // The name of the sending contact
    QString             contactName_;
    // The picture of the sending contact
    QString             contactPicturePath_;
    // The message contents
    ContentsClass       contentsClass_;
    // The message font
    QFont               font_;
    // The message color
    QString             fontColor_;
    // Is the message incoming?
    bool                isIncoming_;
    // The message time
    QDateTime           time_;
    // The message type
    MessageType         type_;
};

#endif
