/***************************************************************************
 *   Copyright (C) 2007-2013 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QLabel>
#include <QToolTip>
#include <QVBoxLayout>
#include <QEvent>
#include <QKeyEvent>
#include <QListWidget>

#include <KDateTable>
#include <KFileDialog>
#include <KIconLoader>
#include <KLocale>

#include "komparatorurllist.h"

#include <version.h>

KomparatorUrlList::KomparatorUrlList( QWidget *parent, const char *name ) :
	KPushButton( parent ),
	m_popup_frame( NULL ), m_list_view( NULL )
{
	Q_UNUSED( name );
	init();
}

KomparatorUrlList::KomparatorUrlList( const QString &text, QWidget *parent, const char *name ) :
	KPushButton( text, parent ),
	m_popup_frame( NULL ), m_list_view( NULL )
{
	Q_UNUSED( name );
	init();
}

KomparatorUrlList::KomparatorUrlList( const KIcon &icon, const QString &text, QWidget *parent, const char *name ) :
	KPushButton( icon, text, parent ),
	m_popup_frame( NULL ), m_list_view( NULL )
{
	Q_UNUSED( name );
	init();
}

KomparatorUrlList::KomparatorUrlList( const KGuiItem &item, QWidget *parent, const char *name ) :
	KPushButton( item, parent ),
	m_popup_frame( NULL ), m_list_view( NULL )
{
	Q_UNUSED( name );
	init();
}

KomparatorUrlList::~KomparatorUrlList()
{
	delete m_list_view;
	m_list_view = NULL;
	delete m_main_box;
	m_main_box = NULL;
	delete m_popup_frame;
	m_popup_frame = NULL;
}

void KomparatorUrlList::init()
{
	setText( i18n( "Virtual subdirectories..." ) );

	m_popup_frame = new KPopupFrame( this );
	m_popup_frame->installEventFilter( this );

	m_main_box = new QWidget( m_popup_frame );
	QHBoxLayout *m_main_box_layout = new QHBoxLayout( m_main_box );
	m_main_box->setLayout( m_main_box_layout );

	m_list_view = new QListWidget( m_main_box );
	m_main_box_layout->addWidget( m_list_view );
	m_list_view->setSelectionMode( QListWidget::SingleSelection );

	QWidget *buttons_box = new QWidget( m_main_box );
	m_main_box_layout->addWidget( buttons_box );
	QVBoxLayout *buttons_box_layout = new QVBoxLayout( buttons_box );
	buttons_box->setLayout( buttons_box_layout );
	m_add_button = new KPushButton( KIcon( SmallIcon( "list-add" ) ), "", buttons_box );
	buttons_box_layout->addWidget( m_add_button );
	m_add_button->setFixedSize( 1.75 * SmallIcon( "edit-copy" ).size() );
	connect( m_add_button, SIGNAL( clicked() ), this, SLOT( slotAdd() ) );
	buttons_box_layout->addWidget( new QLabel( "", buttons_box ) );

	m_remove_button = new KPushButton( KIcon( SmallIcon( "list-remove" ) ), "", buttons_box );
	buttons_box_layout->addWidget( m_remove_button );
	m_remove_button->setFixedSize( 1.75 * SmallIcon( "edit-copy" ).size() );
	connect( m_remove_button, SIGNAL( clicked() ), this, SLOT( slotRemove() ) );
	buttons_box_layout->addWidget( new QLabel( "", buttons_box ) );

	m_clear_button = new KPushButton( KIcon( SmallIcon( "edit-clear" ) ), "", buttons_box );
	buttons_box_layout->addWidget( m_clear_button );
	m_clear_button->setFixedSize( 1.75 * SmallIcon( "edit-copy" ).size() );
	connect( m_clear_button, SIGNAL( clicked() ), this, SLOT( slotClear() ) );

	m_main_box->adjustSize();
	m_popup_frame->setMainWidget( m_main_box );

	const QString add_tt = i18n( "Add an url to the list" );
	m_add_button->setToolTip( add_tt );
	const QString remove_tt = i18n( "Remove the selected url from the list" );
	m_remove_button->setToolTip( remove_tt );
	const QString clear_tt = i18n( "Clears all urls from the list" );
	m_clear_button->setToolTip( clear_tt );

	connect( this, SIGNAL( clicked() ), this, SLOT( slotClicked() ) );
}

void KomparatorUrlList::slotClicked()
{
	if ( m_popup_frame->isHidden() )
	{
		m_main_box->setMinimumWidth( width() );
		m_main_box->setMinimumHeight( height()*5 );
		m_popup_frame->setMainWidget( m_main_box );
		m_popup_frame->popup( mapToGlobal( QPoint( 0, height() ) ) );
	}
}

void KomparatorUrlList::slotAdd()
{
	KFileDialog *dlg = new KFileDialog( KUrl(), "*", (QWidget*) this, NULL );
	dlg->setMode( KFile::Directory );
	if ( dlg->exec() != QDialog::Accepted )
	{
		return;
	}

	new QListWidgetItem( dlg->selectedUrl().url(), m_list_view );
}

void KomparatorUrlList::slotRemove()
{
	delete m_list_view->currentItem();
}

void KomparatorUrlList::slotClear()
{
	clear();
}

void KomparatorUrlList::clear()
{
	m_list_view->clear();
}

KUrl::List KomparatorUrlList::getUrlList()
{
	KUrl::List list;

	int i;
	for ( i=0; i<m_list_view->count(); i++ )
	{
		list.append( KUrl( m_list_view->item( i )->text() ) );
	}

	return list;
}

void KomparatorUrlList::addUrl( const KUrl &url )
{
	new QListWidgetItem( url.url(), m_list_view );
}

int KomparatorUrlList::getUrlNumber()
{
	return m_list_view->count();
}

bool KomparatorUrlList::eventFilter( QObject*, QEvent *e )
{
	if ( e->type() == QEvent::Hide )
	{
		if ( !m_popup_frame->isHidden() ) // This hide event should actually really hide the frame.
		{
			m_popup_frame->hide();
		}

		return true;
	}
	else if ( e->type() == QEvent::KeyRelease || e->type() == QEvent::KeyPress )
	{
		if( ((QKeyEvent *)e)->key() == Qt::Key_Escape ) // Esc would close main window, if it wasn't caught here !!!
		{
			if ( !m_popup_frame->isHidden() ) m_popup_frame->hide();
			return true;
		}
	}

  return false;
}

void KomparatorUrlList::setReadOnly( bool ro )
{
	m_add_button->setEnabled( !ro );
	m_remove_button->setEnabled( !ro );
	m_clear_button->setEnabled( !ro );
}
