/***************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as               *
*   published by the Free Software Foundation; either version 2.0 of      *
*   the License, or (at your option) any later version.                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
*   General Public License for more details.                              *
*                                                                         *
*   You should have received a copy of the GNU General Public             *
*   License along with this program; if not, write to                     *
*   the Free Software Foundation, Inc.,                                   *
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
***************************************************************************/

#include "socket_controller.h"

SocketController::SocketController(char* ip, int port,int f_port, QString password)
    : server_ip(ip), server_port(port), server_port_file(f_port), server_password(password)
{
#ifdef WIN32
    filePath = QDir::toNativeSeparators(QDir::homePath()) + "\\" + KP_DIRECTORY + "\\";
    slash = QString("\\");
#elif defined (linux)
    filePath = QDir::toNativeSeparators(QDir::homePath()) + "/" + KP_DIRECTORY + "/";
    slash = QString("/");
#elif defined __APPLE__
    filePath = QDir::toNativeSeparators(QDir::homePath()) + "/" + KP_DIRECTORY + "/";
    slash = QString("/");
#else
    filePath = QDir::toNativeSeparators(QDir::homePath()) + "/" + KP_DIRECTORY + "/";
    slash = QString("/");
#endif
    file_queue = new QQueue<File*>;
    xml_parser = new XmlParserClient(this);
    connect_to_server(server_ip, server_port, server_port_file, server_password);
}

void SocketController::connect_to_server(char* ip, int port,int f_port, QString password)
{
    master_tcp_client = new auth_tcp_client(password, NULL);
    file_tcp_client = new auth_tcp_client(password, NULL);

    master_tcp_client->connect_to_server(ip, port);
    file_tcp_client->connect_to_server(ip, f_port);
    file_manager = new FileManager(file_tcp_client, this);
    QObject::connect(master_tcp_client, SIGNAL(auth_suceeded()), this, SLOT(auth_succeeded_slot()));
    QObject::connect(master_tcp_client, SIGNAL(received_stream(QString)), this, SLOT(received_stream_slot(QString)));
    QObject::connect(file_tcp_client, SIGNAL(received_stream(QString)), file_manager, SLOT(received_stream_slot_files(QString)));
}

void SocketController::auth_succeeded_slot()
{
    send_to_server(QString(ASK_PARAMETERS));
}

void SocketController::send_to_server(QString msg)
{
    master_tcp_client->write_encrypted_data_to_socket(msg);
}

/*
    Connect the app to the received_stream_signal
    to be aware of reception and handle messages.
*/
void SocketController::received_stream_slot(QString message)
{
    emit update_percent(0, 0);
    if (message.contains(XML_PARAMETERS, Qt::CaseInsensitive)) {
        OsCharacteristic *osCharacteristic = xml_parser->getClientParametersFromXml(message);
        emit client_characteristics(osCharacteristic);
        if (osCharacteristic->system.contains(QString(WINDOWS), Qt::CaseSensitive))
            ask_file_to_server(filePath, osCharacteristic->home_path + "\\" + KP_DIRECTORY, QString(REMOTE_DESKTOP_PNG));
        else
            ask_file_to_server(filePath, osCharacteristic->home_path + "/" + KP_DIRECTORY, QString(REMOTE_DESKTOP_PNG));
    } else if (message.startsWith(QString(QString(XML_DOCTYPE) + QString(XML_FOLDERS) + ">"), Qt::CaseSensitive)) {
        Folder *folderInfo = xml_parser->getDirectoryContent(message);
        QApplication::processEvents();
        emit refresh_directory_view(folderInfo);
    } else if (message.startsWith(QString(QString(XML_DOCTYPE) + QString(XML_FILE_INFO) + ">"), Qt::CaseSensitive)) {
        File *fileInfo = xml_parser->getFileInfo(message);
        QApplication::processEvents();
        emit showMessage(fileInfo->name + tr(" information"), tr("Name : ") + fileInfo->name + tr(" \nExecutable : ") + fileInfo->executable + tr(" \nReadable : ") + fileInfo->readable + tr(" \nSize : ") + fileInfo->size + tr(" \nOwner : ") + fileInfo->owner + tr(" \nExtension : ") + fileInfo->extension + tr(" \nCreation date : ") + fileInfo->creationDate + tr(" \nModification date : ") + fileInfo->modifiedDate, 5000);
        delete fileInfo;
    } else if (message.startsWith(QString(UPLOAD_KO), Qt::CaseSensitive)) {
        // Don't send file.
        File *file = file_queue->dequeue();
        emit remove_from_view(file->name, false);
        process_queue();
        emit showMessage(tr("File error"),tr("Remote computer does not have permissions to receive ") + file->name, 4000);
        delete file;
        emit showLabelSignal(tr(""));
    } else if (message.startsWith(QString(UPLOAD_OK), Qt::CaseSensitive)) {
        // Send file.
        File *file = file_queue->head();
        emit showLabelSignal(tr("Encrypting ") + file->name +  tr(", preparing to upload..."));
        int result = file_manager->send_document(file->name, file->from);
        if (result == 0) {
            File *file = file_queue->dequeue();
            emit remove_from_view(file->name, false);
            process_queue();
            emit showMessage(tr("Error sending file"), file->name + tr(" did not open correctly."), 4000);
            emit showLabelSignal(tr(""));
            delete file;
        }
    } else if (message.startsWith(QString(UPLOAD_SUCCESS), Qt::CaseSensitive)) {
        File *file = file_queue->dequeue();
        emit showMessage(tr("File uploaded"), file->name + tr(" has been uploaded successfully."), 4000);
        emit showLabelSignal(tr(""));
        if (file_queue->isEmpty())
            emit file_uploaded(file);
        process_queue();
        delete file;
    } else if (message.startsWith(QString(UPLOAD_FAILED), Qt::CaseSensitive)) {
        File *file = file_queue->dequeue();
        emit remove_from_view(file->name, false);
        emit showMessage(tr("File uploaded"),file->name + tr(" upload failed.") + file->name, 4000);
        process_queue();
        delete file;
        emit showLabelSignal(tr(""));
    } else if (message.startsWith(QString(QString(XML_DOCTYPE) + QString(XML_FILE_DOWNLOAD_INFO) + ">"), Qt::CaseSensitive)) {
        File *fileInfo = xml_parser->getFileInfo(message);
        QApplication::processEvents();
        fileInfo->to = filePath;
        if (fileInfo->readable.startsWith("true")) {
            if (fileInfo->size.toInt() < FILE_SIZE_LIMIT) { // ~ 60Mo
                if (fileInfo->size.toInt() != 0) {
                    emit showLabelSignal(tr("Downloading ") + fileInfo->name +  tr(" please wait..."));
                    send_to_server(XmlCreatorClient::create_xml_question(fileInfo->directory, fileInfo->name, QString(XML_DOWNLOAD)));
                    file_manager->append_file(fileInfo);
                } else {
                    error_message_dequeue("Remote file size is 0. Nothing to download.");
                    emit remove_from_view(fileInfo->name, true);
                    delete fileInfo;
                }
            } else {
                error_message_dequeue("Please, choose a file smaller than 55MB.");
                emit remove_from_view(fileInfo->name, true);
                delete fileInfo;
            }
        } else {
            error_message_dequeue("Remote file is not readable.");
            emit remove_from_view(fileInfo->name, true);
            delete fileInfo;
        }
    } else if (message.startsWith(QString(XML_COMMAND_BEGIN), Qt::CaseSensitive)) {
        QString result = message.replace(QString(XML_COMMAND_BEGIN), "");
        result = result.replace(QString(XML_COMMAND_END), "");
        result = result.replace(QString("</auth>"), "");
        emit command_results(result);
    }
}

void SocketController::ask_file_to_server(QString to, QString from, QString file_name)
{
    File *file = new File(this);
    file->name = file_name;
    file->is_upload = false;
    file->to = to;
    file->from = from;
    emit showLabelSignal(tr("Queueing ") + file->name + tr(" into file manager's queue..."));

    // If files queue is empty, process queue right after enqueuing.
    // If files queue is not empty. It means a file is currently being downloaded.
    if (file_queue->isEmpty()) {
       file_queue->enqueue(file);
       process_queue();
    } else
        file_queue->enqueue(file);
}

void SocketController::send_file_to_server(QString to, QString from, QString file_name)
{
    File *file = new File(this);
    file->name =file_name;
    file->to = to;
    file->is_upload = true;
    file->from = from;
    QFileInfo *fileInfo = new QFileInfo(file->from + slash +  file->name);

    // Check if file is uploadable.
    if (fileInfo->size() < FILE_SIZE_LIMIT) {
        if (fileInfo->size() != 0) {
            emit showLabelSignal(tr("Queueing ") + file->name + tr(" into file manager's queue..."));
            // If files queue is empty, process queue right after enqueuing.
            // If files queue is not empty. It means a file is currently being downloaded.
            if (file_queue->isEmpty()) {
               file_queue->enqueue(file);
               process_queue();
            } else
                file_queue->enqueue(file);
        } else {
            emit showMessage(tr("File error"),tr("File size is 0. Nothing to download."), 4000);
            emit remove_from_view(file->name, false);
            delete fileInfo;
        }
    } else {
        emit showMessage(tr("File error"),tr("Please, choose a file smaller than 55MB."), 4000);
        emit remove_from_view(file->name, false);
        delete fileInfo;
    }
}

void SocketController::process_queue()
{
    if (!file_queue->isEmpty()) {
        File *file = file_queue->head();
        filePath = file->to;
        if (file->is_upload == false) {
            emit showLabelSignal(tr("Asking ") + file->name +  tr(" to remote computer..."));
            send_to_server(XmlCreatorClient::create_xml_question(file->from, file->name, QString(XML_FILE_DOWNLOAD_INFO)));
        } else {
            emit showLabelSignal(tr("Telling remote computer ") + file->name +  tr(" is coming..."));
            send_to_server(XmlCreatorClient::create_xml_question(file->to, file->name, QString(XML_FILE_UPLOAD_INFO)));
        }
    }
}

void SocketController::executeCommand(QString command, QString path)
{
    send_to_server(XmlCreatorClient::create_xml_question(path, command, QString(XML_COMMAND)));
    emit update_percent(50, 100);
}

void SocketController::error_message_dequeue(QString message)
{
    emit showMessage(tr("File error"),message, 4000);
    emit showLabelSignal(tr(""));
    file_queue->dequeue();
    process_queue();
}
