/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file: gltools.c                                                          */
/*                                                                          */
/*                                                                          */
/* description: interface to gltools-2-4                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"
#include "glrnd.h"
#include "glmesh.h"

/*--------------------------------------------------------------------------*/
/*  vector for storing the additional coordinates, and vector with pointers */
/*  to all these coordinates; the first three have to be set on each        */
/*  element to the element's vertices                                       */
/*--------------------------------------------------------------------------*/

typedef struct gltools_info  GLTOOLS_INFO;
struct gltools_info
{
  const DOF_REAL_VEC   *u;
  const DOF_REAL_D_VEC *ud;
  const DOF_REAL_D_VEC *displacement;
  int          degree;
  int          refinement;
  int          scalar;  /* 0:vector, 1:scalar, 2:vector_norm */

  MESH         *mesh;
  REAL         (*get_est)(EL *);
  double       min, max;

  GLTOOLS_INFO *next;
};

static GLTOOLS_INFO *key_P_info = nil;

static int glrKeyAction_P(glRenderer rnd, int mask)
{
  switch (key_P_info->degree)
  {
  case 0:
  case 1:
      return(GL_FALSE);
  default:
      key_P_info->refinement = 
	(key_P_info->refinement + 1)%(key_P_info->degree+1);
      return(GL_TRUE);
  }
}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
typedef struct cell_info CELL_INFO;
struct cell_info {
  int        dim;
  const REAL *coords[N_VERTICES_MAX];
  const REAL *bary[N_LAMBDA];
  REAL       f[N_VERTICES_MAX];
  int        el_type;
};
static const EL_INFO  *cell_el_info;
static const BAS_FCTS *cell_bas_fcts;
static const REAL     *cell_uh = nil;
static const REAL_D   *cell_uh_d = nil;
static const BAS_FCTS *cell_disp_bas_fcts;
static REAL_D          *cell_displacement = nil;
static int             cell_displacement_size = 0;
static int        cell_f_scalar=1;  /* 0:vector, 1:scalar, 2:vector_norm */
static void       (*cell_coord_to_world)(const EL_INFO *info, const QUAD *quad,
			 int n, const REAL lambda[][N_LAMBDA], REAL_D *world);
static REAL_D     glm_cellflux = {0.0};
static int        cell_count = 0;


static int        const_cell_dofs[N_VERTICES_MAX]={0};

static int        local_cell_dofs[N_VERTICES_MAX]={0,1,2,3};

static const REAL vertex_bary[N_VERTICES_MAX][N_LAMBDA] = {{1.0,0.0,0.0,0.0},
							   {0.0,1.0,0.0,0.0},
							   {0.0,0.0,1.0,0.0},
							   {0.0,0.0,0.0,1.0}};

static const REAL center_bary[4][N_LAMBDA] = {{0.0, 0.0, 0.0, 0.0},
					      {0.5, 0.5, 0.0, 0.0},
					      {1.0/3.0, 1.0/3.0, 1.0/3.0, 0.0},
					      {0.25,0.25,0.25,0.25}};

static int child_vertex_2d[2][3] = {{2, 0, -1},{1, 2, -1}};

#define STIP_WIDTH 0.2
/* plot two triangles for each 1D simplex */
static void simplex_1d_2d(glMesh m, int cell_count, REAL *f, int nf, 
			  REAL **coords, glmSimplexCallback sxf)
{
  FUNCNAME("simplex_1d_2d");
    
  static REAL coords_1d_2d[4][2]={{0.0,0.0},{1.0,0.0},
				  {0.0,STIP_WIDTH},{1.0,STIP_WIDTH}};
  static REAL *cell_coords_1d_2d[2][3]
    ={{coords_1d_2d[1],coords_1d_2d[2],coords_1d_2d[0]},
      {coords_1d_2d[2],coords_1d_2d[1],coords_1d_2d[3]}};
  static int local_cell_dofs_1d_2d[2][3]={{1,0,0},{0,1,1}};
  static int const_cell_dofs_1d_2d[3]={0,0,0};

  DEBUG_TEST_EXIT(coords, "no coords");

#if DIM_OF_WORLD == 1  
  coords_1d_2d[0][0] = coords_1d_2d[2][0] = coords[0][0];
  coords_1d_2d[1][0] = coords_1d_2d[3][0] = coords[1][0];
#else
  {
    REAL len, normal[2];

    normal[0] = coords[1][1] - coords[0][1];
    normal[1] = coords[0][0] - coords[1][0];

    len = STIP_WIDTH * 0.3 / (sqrt(SQR(normal[0])+SQR(normal[1])));
    normal[0] *= len;
    normal[1] *= len;

    coords_1d_2d[0][0] = coords[0][0];
    coords_1d_2d[0][1] = coords[0][1];
    coords_1d_2d[1][0] = coords[1][0];
    coords_1d_2d[1][1] = coords[1][1];

    coords_1d_2d[2][0] = coords[0][0] + normal[0];
    coords_1d_2d[2][1] = coords[0][1] + normal[1];
    coords_1d_2d[3][0] = coords[1][0] + normal[0];
    coords_1d_2d[3][1] = coords[1][1] + normal[1];
  }
#endif

  if (nf >= 3) 
  {
    sxf(m, cell_count, 1, f, local_cell_dofs_1d_2d[0], cell_coords_1d_2d[0]);
    sxf(m, cell_count, 1, f, local_cell_dofs_1d_2d[1], cell_coords_1d_2d[1]);
  } 
  else
  {
    sxf(m, cell_count, 1, f, const_cell_dofs_1d_2d, cell_coords_1d_2d[0]);
    sxf(m, cell_count, 1, f, const_cell_dofs_1d_2d, cell_coords_1d_2d[1]);
  }
  
  return;
}

static void recursive_cell(glMesh m, glmSimplexCallback sxf, int level,
			   CELL_INFO *cell_info)
{
  if (level <= 0) 
  {
    if(cell_info->dim == 1) 
      simplex_1d_2d(m, cell_count, cell_info->f, 3,
		    (REAL **)cell_info->coords, sxf);
    else
      if (cell_f_scalar) 
	sxf(m, cell_count, 1, cell_info->f,
	    local_cell_dofs, (REAL **)cell_info->coords);
      else 
	{
	  eval_uh_d(center_bary[cell_info->dim],
		    cell_uh_d, cell_bas_fcts, glm_cellflux);
	  sxf(m, 0, 1, glm_cellflux, const_cell_dofs,
	      (REAL **)cell_info->coords);
	}
  }
  else
  {
    CELL_INFO  child_info;
    REAL_D     new_coords;
    REAL       new_bary[1][N_LAMBDA], new_f;
    int        i, ichild;
    
    for (i = 0; i <= cell_info->dim; i++)
      new_bary[0][i] = 0.5*(cell_info->bary[0][i] + cell_info->bary[1][i]);
    
    if (cell_coord_to_world) {
      cell_coord_to_world(cell_el_info, nil, 1, 
			  (const REAL (*)[N_LAMBDA])new_bary, &new_coords);
    }
    else 
    {
      if (0 && cell_disp_bas_fcts)
      {
	eval_uh_d(new_bary[0], (const REAL_D *)cell_displacement,
		  cell_disp_bas_fcts, new_coords);
      }
      else
      {
	for (i = 0; i <= cell_info->dim; i++)
	  new_coords[i] = 0.5*(cell_info->coords[0][i] 
			       + cell_info->coords[1][i]);
      }
    }
    switch (cell_f_scalar) 
    {
    case 1:
      new_f = eval_uh(new_bary[0], cell_uh, cell_bas_fcts);
      break;
    case 2:
    {
      REAL_D f_d;
      eval_uh_d(new_bary[0], cell_uh_d, cell_bas_fcts, f_d);
      new_f = NORM_DOW(f_d);
      break;
    }
    default:
      new_f = 0.0;
    }

    child_info.dim = cell_info->dim;

    for (ichild = 0; ichild < 2; ++ichild)
    {
      switch(cell_info->dim) {
      case 1:
	child_info.coords[ichild]   = cell_info->coords[ichild];
	child_info.coords[1-ichild] = new_coords;
	child_info.bary[ichild]     = cell_info->bary[ichild];
	child_info.bary[1-ichild]   = new_bary[0];
	child_info.f[ichild]        = cell_info->f[ichild];
	child_info.f[1-ichild]      = new_f;
	break;
      case 2:
	for (i = 0; i < 2; i++) 
	  {
	    int j = child_vertex_2d[ichild][i];
	    child_info.coords[i] = cell_info->coords[j];
	    child_info.bary[i]   = cell_info->bary[j];
	    child_info.f[i]      = cell_info->f[j];
	  }
	child_info.coords[2] = new_coords;
	child_info.bary[2]   = new_bary[0];
	child_info.f[2]      = new_f;
	break;
      case 3:
	for (i = 0; i < 3; i++) 
	  {
	    int j = child_vertex_3d[cell_info->el_type][ichild][i];
	    child_info.coords[i] = cell_info->coords[j];
	    child_info.bary[i]   = cell_info->bary[j];
	    child_info.f[i]      = cell_info->f[j];
	  }
	child_info.coords[3] = new_coords;
	child_info.bary[3]   = new_bary[0];
	child_info.f[3]      = new_f;
	child_info.el_type   = (cell_info->el_type + 1) % 3;
      }

      recursive_cell(m, sxf, level-1, &child_info);
    }
  }
  return;
}

static void m_cell_loop(glMesh m, void *data, glmSimplexCallback sxf)
{
  FUNCNAME("m_cell_loop");
  CELL_INFO        cell_info[1];
  GLTOOLS_INFO     *info = (GLTOOLS_INFO *) data;
  const DOF_REAL_VEC     *u  = NULL;
  const DOF_REAL_D_VEC   *ud = NULL;
  MESH             *mesh;
  PARAMETRIC       *parametric;
  const DOF_ADMIN  *admin;
  TRAVERSE_STACK   *stack = get_traverse_stack();
  const EL_INFO    *el_info;
  REAL_D           wc[N_VERTICES_MAX];
  int              i, j, dim;

  
  cell_f_scalar = info->scalar;

  if (cell_f_scalar==1) 
  {
    u = info->u;
    if (!u)
    {
      ERROR("no dof_real_vec\n");
      return;
    }
    mesh = u->fe_space->mesh;
    admin = u->fe_space->admin;
    cell_bas_fcts = u->fe_space->bas_fcts;
  }
  else
  {
    ud = info->ud;
    if (!ud)
    {
      ERROR("no dof_real_d_vec\n");
      return;
    }
    mesh = ud->fe_space->mesh;
    admin = ud->fe_space->admin;
    cell_bas_fcts = ud->fe_space->bas_fcts;
  }
  parametric = mesh->parametric;
  dim = mesh->dim;
  
  if (info->displacement) 
  {
      cell_disp_bas_fcts = info->displacement->fe_space->bas_fcts;
  }
  else
  {
      cell_disp_bas_fcts = nil;
  }
  cell_count=1;

  for (el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL|FILL_COORDS);
       el_info;
       el_info=traverse_next(stack,el_info))
  {
    cell_el_info = el_info;
    if (cell_f_scalar==1)
    {
      cell_uh = cell_bas_fcts->get_real_vec(el_info->el, u, nil);
    }
    else
    {
      cell_uh_d = cell_bas_fcts->get_real_d_vec(el_info->el, ud, nil);
    }

    cell_info->dim = dim;

    /* correct element coordinates for parametric elements */
    if (parametric) {
      parametric->init_element(el_info, parametric);
      cell_coord_to_world = parametric->coord_to_world;
      cell_coord_to_world(el_info, nil, N_VERTICES(dim), vertex_bary, wc);
      for(i = 0; i < N_VERTICES(dim);i++)
	cell_info->coords[i] = wc[i];
    }
    else {
      cell_coord_to_world = nil;
      if (info->displacement) {  /* add displacements */
	if (cell_displacement_size < cell_disp_bas_fcts->n_bas_fcts) {
	  cell_displacement = 
	    MEM_REALLOC(cell_displacement, cell_displacement_size,
			cell_disp_bas_fcts->n_bas_fcts, REAL_D);
	}
	cell_disp_bas_fcts->get_real_d_vec(el_info->el, info->displacement,
					   cell_displacement);
	for(i = 0; i < N_VERTICES(dim);i++) {
	  eval_uh_d(vertex_bary[i], (const REAL_D *)cell_displacement, 
		    cell_disp_bas_fcts, wc[i]);
	  for (j=0; j<DIM_OF_WORLD; ++j)
	    wc[i][j] += el_info->coord[i][j];
	  cell_info->coords[i] = wc[i];
	}
      }
      else
      {
	cell_coord_to_world = nil;
	for(i = 0; i < N_VERTICES(dim);i++)
	  cell_info->coords[i] = el_info->coord[i];
      }
    }
    for (i = 0; i < N_VERTICES(dim);i++)
	cell_info->bary[i] = vertex_bary[i];
    
    if (cell_f_scalar==1) 
    {
      for (i = 0; i < N_VERTICES(dim);i++)
	cell_info->f[i] = eval_uh(vertex_bary[i], cell_uh, cell_bas_fcts);
    }
    else if (cell_f_scalar==2) 
    {
      REAL_D f_d;
      for (i = 0; i < N_VERTICES(dim);i++) 
      {
	eval_uh_d(vertex_bary[i], cell_uh_d, cell_bas_fcts, f_d);
	cell_info->f[i] = NORM_DOW(f_d);
      }
    }

    if(dim == 3)
      cell_info->el_type = el_info->el_type;

    if (info->refinement)
      recursive_cell(m, sxf, dim*info->refinement, cell_info);
    else
      recursive_cell(m, sxf, 0, cell_info);

    ++cell_count;
  }

  free_traverse_stack(stack);

  return;
}


static int next_dialog = 1;

/* Next two functions meant for external add-ons, s.t. those can get
 * hold of the dialog switch.
 */
int gltools_get_next_dialog(void)
{
  return next_dialog;
}

void gltools_set_next_dialog(int dialog)
{
  next_dialog = dialog;
}

static int glrKeyAction_Q(glRenderer rnd, int mask)
{
  int   dialog;
  glrGetDialog(rnd, &dialog);
  glrSetDialog(rnd, !dialog);
  next_dialog = 0;
  return GL_TRUE;
}

static int glrKeyAction_q(glRenderer rnd, int mask)
{
  int   dialog;
  glrGetDialog(rnd, &dialog);
  glrSetDialog(rnd, !dialog);
  next_dialog = 1;
  return GL_TRUE;
}

static int glrKeyAction_s(glRenderer rnd, int mask)
{
  glrSetDialog(rnd, 1);
  next_dialog = 1;
  return GL_TRUE;
}

static int glrKeyAction_X(glRenderer rnd, int mask)
{
  exit(0);

  return GL_TRUE;
}

static void xmin_xmax(MESH *mesh, REAL_D xmin, REAL_D xmax)
{
  PARAMETRIC *parametric;
  EL_INFO    mel_info[1];
  MACRO_EL   *mel;
  int        m, n, i, dim = mesh->dim;

  if (!mesh) return;
  parametric = mesh->parametric;
  mel_info->fill_flag = FILL_COORDS;

  if(parametric)
    mel_info->fill_flag |= FILL_PROJECTION;

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    xmin[n] = LARGE;
    xmax[n] = -LARGE;
  }

  for (m = 0; m < mesh->n_macro_el; m++) {
    mel = mesh->macro_els + m;

    fill_macro_info(mesh, mel, mel_info);
    /* correct element coordinates for parametric elements */
    if (parametric) {
      parametric->init_element(mel_info, parametric);
      parametric->coord_to_world(mel_info, nil, N_VERTICES(dim),
				 vertex_bary, mel_info->coord);
    }
    
    for (n = 0; n < DIM_OF_WORLD; n++) {
      for (i=0; i<N_VERTICES(dim); i++) {
	xmin[n] = MIN(xmin[n], mel_info->coord[i][n]);
	xmax[n] = MAX(xmax[n], mel_info->coord[i][n]);
      }
    }
  }
  return;
}

GLTOOLS_WINDOW open_gltools_window(const char *title, const char *geometry,
				   const REAL *world, MESH *mesh, int dialog)
{
  FUNCNAME("open_gltools_window");
  REAL             xmin[3], xmax[3];
  glRenderer       rnd;
  glRendererState  state;
  int              x = 0, y = 0, w = 300, h = 300, dim = mesh->dim;
  const char       *s;

  if (dim<=0) {
    ERROR("Sorry, only implemented for dim > 0.\n");
    return(nil);
  } 

  if (dim==1 && DIM_OF_WORLD==3) {
    ERROR("Sorry, not implemented for dim==1 && DIM_OF_WORLD==3.\n");
    return(nil);
  } 

  INFO(0,2,"\n");
  if (geometry)
  {
    if (strchr(geometry, 'x'))
    {
      if ((s = strchr(geometry, '+')))
      {
	if (strchr(s+1, '+'))   /*  "wwxhh+xx+yy"   */
	  sscanf(geometry, "%dx%d+%d+%d", &w, &h, &x, &y);
	else                     /*  "wwxhh+xx"      */
	  sscanf(geometry, "%dx%d+%d", &w, &h, &x);
      }
      else                       /*  "wwxhh"         */
      {
	sscanf(geometry, "%dx%d", &w, &h);
      }
    }
    else
    {
      if ((s = strchr(geometry, '+')))
      {
	if (strchr(s+1, '+'))    /*  "xx+yy"         */
	  sscanf(s, "+%d+%d", &x, &y);
	else                     /*  "wwxhh+xx"      */
	  sscanf(s, "+%d", &x);
      }
    }
  }
  if (!title)
    title = "ALBERTA";
  rnd = glrCreate((char *) title, x, y, w, h);
  glrSetDialog(rnd, dialog);
  state = glrGetRendererState(rnd);
  state->sensitivity *= M_SQRT2;

  glrRegisterKeyAction(rnd, GLW_Q, glrKeyAction_Q,"Q: quit dialog");
  glrRegisterKeyAction(rnd, GLW_q, glrKeyAction_q,"q: quit");
  glrRegisterKeyAction(rnd, GLW_s, glrKeyAction_s,"s: enter dialog");
  glrRegisterKeyAction(rnd, GLW_X, glrKeyAction_X,"X: exit program");

  if (world)
  {
    xmin[0] = world[0];
    xmax[0] = world[1];
#if DIM_OF_WORLD == 2
    xmin[1] = world[2];
    xmax[1] = world[3];
#if DIM_OF_WORLD == 3
    xmin[2] = world[4];
    xmax[2] = world[5];
#endif
#endif
  }
  else if (mesh)
  {
    xmin_xmax(mesh, xmin, xmax);
  }
  else
  {
    xmin[0] = 0.0;
    xmax[0] = 1.0;
    xmin[1] = 0.0;
    xmax[1] = 1.0;
    xmin[2] = 0.0;
    xmax[2] = 1.0;
  }

  if (dim<DIM_OF_WORLD && dim==1) {
    xmin[0] -= STIP_WIDTH*0.5; 
    xmin[1] -= STIP_WIDTH*0.5; 
    xmax[0] += STIP_WIDTH*0.5; 
    xmax[1] += STIP_WIDTH*0.5; 
  }

#if DIM_OF_WORLD == 1
  glrSetAxisName(rnd, 0, "x");
  glrSetAxisName(rnd, 2, "f(x)");
  xmin[1] = 0.0;
  xmax[1] = STIP_WIDTH;
  xmin[2] = -1.0;
  xmax[2] = 1.0;
#elif DIM_OF_WORLD == 2
  glrSetAxisName(rnd, 0, "x");
  glrSetAxisName(rnd, 1, "y");
  glrSetAxisName(rnd, 2, "f(x,y)");
  xmin[2] = -1.0;
  xmax[2] = 1.0;
#elif DIM_OF_WORLD == 3
  glrSetAxisName(rnd, 0, "x");
  glrSetAxisName(rnd, 1, "y");
  glrSetAxisName(rnd, 2, "z");
#endif

  glrSetVolume(rnd, xmin[0], xmax[0], xmin[1], xmax[1], xmin[2], xmax[2]);

  return((GLTOOLS_WINDOW)  rnd);
}

void close_gltools_window(GLTOOLS_WINDOW win)
{
  if (win)
    glrDestroy((glRenderer) win);
  return;
}

void gltools_disp_drv(GLTOOLS_WINDOW win, const DOF_REAL_VEC *u,
		      REAL min, REAL max, const DOF_REAL_D_VEC *disp)
{
  FUNCNAME("gltools_disp_drv");
  static GLTOOLS_INFO *first = nil;
  GLTOOLS_INFO        *info;
  MESH                *mesh;
  int                 degree;
  int                 ndof;
  glMesh              glm;
  glRenderer          rnd;

  INFO(0,2,"\n");
  if (!u || !win)  return;

  rnd = (glRenderer) win;
  glrGetDialog(rnd, &next_dialog);

  mesh       = u->fe_space->mesh;
  degree     = u->fe_space->bas_fcts->degree;
  ndof       = u->fe_space->admin->size_used;

#if 0
  if(mesh->dim != DIM_OF_WORLD) {
    ERROR("Sorry, only implemented for dim == DIM_OF_WORLD.\n");
    return;
  } 
#endif

  for (info = first; info; info = info->next)
    if (info->u == u)  break;

  if (!info)
  {
    info = MEM_CALLOC(1, GLTOOLS_INFO);
    info->degree      = degree;
    info->refinement  = 0;
    info->u           = u;
    info->scalar      = 1;

    info->next = first;
    first = info;
  }
  info->displacement = disp;

  glm = glmCreate(ndof,
		  mesh->n_elements,
		  MAX(2,DIM_OF_WORLD),
		  info,m_cell_loop);

  glrRegisterKeyAction(rnd, GLW_P, glrKeyAction_P,"P: use Lagrange grid");
  key_P_info = info;

  if (u->name)
    glrSetUserInfo(rnd, "%s", u->name);

  glmSetVoffset(glm, 0);
  glrSetInfoCallback(rnd, (glrDrawCallback) glmDrawInfo);

  if (ABS(min - max) < 1.e-25) min++;
  glmSetFunction(glm, u->vec, min, max);
  glRender(rnd, (glrDrawCallback)(glmDraw), glm);
  glmDestroy(glm);

  key_P_info = nil;
  glrRegisterKeyAction(rnd, GLW_P, nil, nil);
  glrSetDialog(rnd, next_dialog);

  return;
}

void gltools_drv(GLTOOLS_WINDOW win, const DOF_REAL_VEC *u,
		 REAL min, REAL max)
{
  gltools_disp_drv(win, u, min, max, nil);
}

void gltools_disp_drv_d(GLTOOLS_WINDOW win, const DOF_REAL_D_VEC *u, 
		   REAL min, REAL max, const DOF_REAL_D_VEC *disp)
{
  FUNCNAME("gltools_disp_drv_d");
  static GLTOOLS_INFO *first = nil;
  GLTOOLS_INFO        *info;
  MESH                *mesh;
  int                 degree;
  int                 ndof;
  glMesh              glm;
  glRenderer          rnd;

  INFO(0,2,"\n");
  if (!u || !win)  return;

  rnd = (glRenderer) win;
  glrGetDialog(rnd, &next_dialog);

  mesh       = u->fe_space->mesh;
  degree     = u->fe_space->bas_fcts->degree;
  ndof       = u->fe_space->admin->size_used;

  if(mesh->dim != DIM_OF_WORLD) {
    ERROR("Sorry, only implemented for dim == DIM_OF_WORLD.\n");
    return;
  } 

  for (info = first; info; info = info->next)
    if (info->ud == u)  break;

  if (!info)
  {
    info = MEM_CALLOC(1, GLTOOLS_INFO);
    info->degree      = degree;
    info->refinement  = 0;
    info->ud          = u;
    info->scalar      = 2;

    info->next = first;
    first = info;
  }
  info->displacement = disp;

  glm = glmCreate(ndof,
		  mesh->n_elements,
		  MAX(2,DIM_OF_WORLD),
		  info, m_cell_loop);

  glrRegisterKeyAction(rnd, GLW_P, glrKeyAction_P,"P: use Lagrange grid");
  key_P_info = info;

  if (u->name)
    glrSetUserInfo(rnd, "%s", u->name);

  glmSetVoffset(glm, 0);
  glrSetInfoCallback(rnd, (glrDrawCallback) glmDrawInfo);

  if (min >= max) {
      min = dof_min_d(u);
      max = dof_max_d(u);
  }
  glmSetFunction(glm, (REAL *)u->vec, min, max);
  glRender(rnd, (glrDrawCallback)(glmDraw), glm);
  glmDestroy(glm);

  key_P_info = nil;
  glrRegisterKeyAction(rnd, GLW_P, nil, nil);
  glrSetDialog(rnd, next_dialog);

  return;
}

void gltools_drv_d(GLTOOLS_WINDOW win, const DOF_REAL_D_VEC *u, 
		   REAL min, REAL max)
{
  gltools_disp_drv_d(win, u, min, max, nil);
}

void gltools_disp_vec(GLTOOLS_WINDOW win, const DOF_REAL_D_VEC *u, 
		      REAL min, REAL max, const DOF_REAL_D_VEC *disp)
{
  FUNCNAME("gltools_disp_vec");
  GLTOOLS_INFO        info[1] = {{0}};
  MESH                *mesh;
  int                 degree;
  int                 ndof;
  glMesh              glm;
  glRenderer          rnd;

  INFO(0,2,"\n");
  if (!u || !win)  return;

  rnd = (glRenderer) win;
  glrGetDialog(rnd, &next_dialog);

  mesh       = u->fe_space->mesh;
  degree     = u->fe_space->bas_fcts->degree;
  ndof       = u->fe_space->admin->size_used;

  if(mesh->dim != DIM_OF_WORLD) {
    ERROR("Sorry, only implemented for dim == DIM_OF_WORLD.\n");
    return;
  } 

  info->degree      = degree;
  info->refinement  = 0;
  info->ud          = u;
  info->scalar      = 0;
  info->displacement = disp;

  glm = glmCreate(1,
		  1,
		  MAX(2,DIM_OF_WORLD),
		  info, m_cell_loop);

  glrRegisterKeyAction(rnd, GLW_P, glrKeyAction_P,"P: use Lagrange grid");
  key_P_info = info;

  if (u->name)
    glrSetUserInfo(rnd, "%s", u->name);

  glmSetVoffset(glm, 0);
  glrSetInfoCallback(rnd, (glrDrawCallback) glmDrawInfo);

  if (min >= max) {
      min = dof_min_d(u);
      max = dof_max_d(u);
  }
/*   glmSetFunction(glm, (REAL *)u->vec, min, max); */
  glmSetCellFlux(glm, glm_cellflux, min, max);
  glRender(rnd, (glrDrawCallback)(glmDraw), glm);
  glmDestroy(glm);

  key_P_info = nil;
  glrRegisterKeyAction(rnd, GLW_P, nil, nil);
  glrSetDialog(rnd, next_dialog);

  return;
}

void gltools_vec(GLTOOLS_WINDOW win, const DOF_REAL_D_VEC *u, 
		 REAL min, REAL max)
{
  gltools_disp_vec(win, u, min, max, nil);
}

static void est_loop(glMesh m, void *data, glmSimplexCallback sxf)
{
  GLTOOLS_INFO     *info = (GLTOOLS_INFO *)data;
  MESH             *mesh = info->mesh;
  PARAMETRIC       *parametric = mesh->parametric;
  int              i, j, ic = 0, dim = mesh->dim;
  const EL_INFO    *el_info;
  REAL             *coord[N_VERTICES_MAX], el_val;
  REAL_D           wc[N_VERTICES_MAX];
  TRAVERSE_STACK   *stack = get_traverse_stack();
  int              dof0[N_VERTICES_MAX]={0};

  if (info->displacement)
    cell_disp_bas_fcts = info->displacement->fe_space->bas_fcts;
  else
    cell_disp_bas_fcts = nil;

  el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL|FILL_COORDS);
  do
  {
    /* correct element coordinates for parametric elements */
    if (parametric) {
      parametric->init_element(el_info, parametric);
      parametric->coord_to_world(el_info, nil, N_VERTICES(dim),
				 vertex_bary, wc);
    }
    else
    {
      if (info->displacement)   /* add displacements */
      {
	if (cell_displacement_size < cell_disp_bas_fcts->n_bas_fcts) 
	{
	  cell_displacement = 
	    MEM_REALLOC(cell_displacement, cell_displacement_size,
			cell_disp_bas_fcts->n_bas_fcts, REAL_D);
	}
	cell_disp_bas_fcts->get_real_d_vec(el_info->el, info->displacement,
					   cell_displacement);
	for(i = 0; i < N_VERTICES(dim);i++)
	{
	  eval_uh_d(vertex_bary[i], (const REAL_D *)cell_displacement, 
		    cell_disp_bas_fcts, wc[i]);
	  for (j=0; j<DIM_OF_WORLD; ++j)
	    wc[i][j] += el_info->coord[i][j];
	}
      } 
      else 
      {
	for (i = 0; i < N_VERTICES(dim); i++)
	  COPY_DOW(el_info->coord[i], wc[i]);
      }
    }

    for (i = 0; i < N_VERTICES(dim); i++)
      coord[i] = wc[i];

    el_val = info->get_est(el_info->el);

    if(dim==1)
      simplex_1d_2d(m, ic+1, &el_val, 1, coord, sxf);
    else
      sxf(m, ic+1, 1, &el_val, dof0, coord);

    ic++;
  } while ((el_info = traverse_next(stack,el_info)));

  free_traverse_stack(stack);
}

void gltools_disp_est(GLTOOLS_WINDOW win, MESH *mesh, REAL (*get_est)(EL *),
		 REAL min, REAL max, const DOF_REAL_D_VEC *disp)
{
  FUNCNAME("gltools_est");
  static GLTOOLS_INFO gltools_info ={0};
  glMesh          glm;
  glRenderer      rnd;
  int             n;
  static REAL     val=0.0;
  const EL_INFO   *el_info;
  TRAVERSE_STACK  *stack;

  INFO(0,2,"\n");

  if (!mesh || !win || !get_est)  return;

  rnd = (glRenderer) win;
  glrGetDialog(rnd, &next_dialog);

  gltools_info.mesh = mesh;
  gltools_info.get_est = get_est;
  gltools_info.displacement = disp;

  if(mesh->dim != DIM_OF_WORLD) {
    ERROR("Sorry, only implemented for dim == DIM_OF_WORLD.\n");
    return;
  } 

  glm=glmCreate(mesh->n_elements, mesh->n_elements, MAX(2,DIM_OF_WORLD),
		&gltools_info, est_loop);
  glmSetVoffset(glm,0);
  glrSetInfoCallback(rnd,(glrDrawCallback)glmDrawInfo);

  if (min >= max) 
  {
    min = LARGE;
    max = -LARGE;
    stack = get_traverse_stack();
    el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL);
    n = 0;
    while (el_info)
    {
      REAL  est = (*get_est)(el_info->el);
      min = MIN(min, est);
      max = MAX(max, est);
      el_info = traverse_next(stack,el_info);
    }
    free_traverse_stack(stack);
  }

  if (min > max) min = max = 0.0;

  gltools_info.min = min;
  gltools_info.max = max;
  glmSetFunction(glm, &val, min, max);
  glRender(rnd, (glrDrawCallback)(glmDraw), glm);
  glmDestroy(glm);
  glrSetDialog(rnd, next_dialog);

  return;
}


void gltools_est(GLTOOLS_WINDOW win, MESH *mesh, REAL (*get_est)(EL *),
		 REAL min, REAL max)
{
  /* FUNCNAME("gltools_est"); */
  gltools_disp_est(win, mesh, get_est, min, max, nil);
}


struct mesh_loop_data
{
    MESH  *mesh;
    int   mark;
    const DOF_REAL_D_VEC *displacement;
};

static void mesh_loop(glMesh m, void *ud, glmSimplexCallback sxf)
{
  struct mesh_loop_data *data = (struct mesh_loop_data *)ud;
  PARAMETRIC       *parametric = data->mesh->parametric;
  int              i, j, ic = 1, dim = data->mesh->dim;
  const EL_INFO    *el_info;
  REAL             *coord[N_VERTICES_MAX];
  REAL             val = 0.0;
  REAL_D           wc[N_VERTICES_MAX];
  TRAVERSE_STACK   *stack = get_traverse_stack();
  const REAL_D     *el_displacement;
  const BAS_FCTS   *disp_bas_fcts=nil;
  static int       dof[N_VERTICES_MAX]={0};
  
  if (data->displacement) {
      disp_bas_fcts = data->displacement->fe_space->bas_fcts;
  }

  el_info = traverse_first(stack, data->mesh, -1, CALL_LEAF_EL|FILL_COORDS);
  do
  {
    if (data->mark)
    {
      if (el_info->el->mark > 0)        val =  1.0;
      else if (el_info->el->mark < 0)   val = -1.0;
      else                              val =  0.0;
    }

    /* correct element coordinates for parametric elements */
    if (parametric) {
      parametric->init_element(el_info, parametric);
      parametric->coord_to_world(el_info, nil, N_VERTICES(dim),
				 vertex_bary, wc);
    }
    else
    {
	if (data->displacement) {
	    el_displacement
		= disp_bas_fcts->get_real_d_vec(el_info->el,
						data->displacement, nil);
	    for(i = 0; i < N_VERTICES(dim);i++) {
		eval_uh_d(vertex_bary[i], el_displacement, 
			  disp_bas_fcts, wc[i]);
		for (j=0; j<DIM_OF_WORLD; ++j)
		    wc[i][j] += el_info->coord[i][j];
	    }
	} else {
	    for (i = 0; i < N_VERTICES(dim); i++)
		COPY_DOW(el_info->coord[i], wc[i]);
	}
    }

    for(i = 0; i < N_VERTICES(dim); i++)
      coord[i] = wc[i];

    if(dim==1)
      simplex_1d_2d(m, ic++, &val, 1, coord, sxf);
    else
      sxf(m, ic++, 1, &val, dof, coord);

 } while ((el_info = traverse_next(stack,el_info)));

  free_traverse_stack(stack);
}

void gltools_mesh(GLTOOLS_WINDOW win, MESH *mesh, int mark)
{
  FUNCNAME("gltools_mesh");
  struct mesh_loop_data data[1];
  glMesh          glm;
  glRenderer      rnd;
  static REAL     val=0.0;

  INFO(0,2,"\n");

  if (!(data->mesh = mesh) || !win)  return;

  if(mesh->dim != DIM_OF_WORLD) {
    ERROR("Sorry, only implemented for dim == DIM_OF_WORLD.\n");
    return;
  } 

  data->displacement = nil;

  rnd = (glRenderer) win;
  glrGetDialog(rnd, &next_dialog);

  if ((data->mark = mark))
  {
    glm=glmCreate(1, mesh->n_elements, MAX(2,DIM_OF_WORLD), data, mesh_loop);
    glmSetVoffset(glm,0);
    glrSetInfoCallback(rnd,(glrDrawCallback)glmDrawInfo);
    glmSetFunction(glm, &val, -1.0, 1.0);
  }
  else
  {
    glm=glmCreate(1, mesh->n_elements, MAX(2,DIM_OF_WORLD), data, mesh_loop);
    glmSetVoffset(glm,0);
    glrSetInfoCallback(rnd,(glrDrawCallback)glmDrawInfo);
    glmSetFunction(glm, &val, 0.0, 1.0);
  }
  glRender(rnd, (glrDrawCallback)(glmDraw), glm);
  glmDestroy(glm);
  glrSetDialog(rnd, next_dialog);

  return;
}

void gltools_disp_mesh(GLTOOLS_WINDOW win, MESH *mesh, int mark,
		       const DOF_REAL_D_VEC *disp)
{
  FUNCNAME("gltools_disp_mesh");
  struct mesh_loop_data data[1];
  glMesh          glm;
  glRenderer      rnd;
  static REAL     val=0.0;

  INFO(0,2,"\n");

  if (!(data->mesh = mesh) || !win)  return;

  if(mesh->dim != DIM_OF_WORLD) {
    ERROR("Sorry, only implemented for dim == DIM_OF_WORLD.\n");
    return;
  } 

  data->displacement = disp;

  rnd = (glRenderer) win;
  glrGetDialog(rnd, &next_dialog);

  if ((data->mark = mark))
  {
    glm=glmCreate(1, mesh->n_elements, MAX(2,DIM_OF_WORLD), data, mesh_loop);
    glmSetVoffset(glm,0);
    glrSetInfoCallback(rnd,(glrDrawCallback)glmDrawInfo);
    glmSetFunction(glm, &val, -1.0, 1.0);
  }
  else
  {
    glm=glmCreate(1, mesh->n_elements, MAX(2,DIM_OF_WORLD), data, mesh_loop);
    glmSetVoffset(glm,0);
    glrSetInfoCallback(rnd,(glrDrawCallback)glmDrawInfo);
    glmSetFunction(glm, &val, 0.0, 1.0);
  }
  glRender(rnd, (glrDrawCallback)(glmDraw), glm);
  glmDestroy(glm);
  glrSetDialog(rnd, next_dialog);

  return;
}
