/*
 * Copyright (c) 2001, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jdi.ClassPrepareRequest.addClassFilter_s;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

import com.sun.jdi.*;
import com.sun.jdi.event.*;
import com.sun.jdi.request.*;

import java.util.*;
import java.io.*;

/**
 * The test for the implementation of an object of the type
 * ClassPrepareRequest.
 *
 * The test checks that results of the method
 * <code>com.sun.jdi.ClassPrepareRequest.addClassFilter(String)</code>
 * complies with its spec.
 *
 * The test checks up on the following assertion:
 *    Restricts the events generated by this request to those
 *    whose location is in a class
 *    whose name matches a restricted regular expression.
 * The cases to check include both a pattern that begin with '*'
 * one that end with '*'.
 *
 * The test works as follows.
 * - The debugger
 *   - creates two class filter objects,
 *     first one  using a pattern that begin with '*', and
 *     second one using a pattern that end with '*'
 *     to filter ClassPrepareEvents in two debuggee's  tested threads,
 *   - sets up two ClassPrepareRequests,
 *   - resumes the debuggee, and
 *   - waits for expected ClassPrepareEvents.
 * - The debuggee creates and starts two threads
 *   whose 'run' methods create objects of ClassTypes needed
 *   to generate Events and to test the filters.
 * - Upon getting the events, the debugger performs the checks required.
 */

public class filter_s001 extends TestDebuggerType1 {

    public static void main (String argv[]) {
        int result = run(argv,System.out);
        if (result != 0) {
            throw new RuntimeException("TEST FAILED with result " + result);
        }
    }

    public static int run (String argv[], PrintStream out) {
        debuggeeName = "nsk.jdi.ClassPrepareRequest.addClassFilter_s.filter_s001a";
        return new filter_s001().runThis(argv, out);
    }

    private String testedClassName1 = "*TestClass11";
    private String testedClassName2 =
        "nsk.jdi.ClassPrepareRequest.addClassFilter_s.Thread2filter_s001a*";

    private String className1 = "TestClass11";
    private String className2 =
        "nsk.jdi.ClassPrepareRequest.addClassFilter_s.Thread2filter_s001a";

    protected void testRun() {
        EventRequest  eventRequest1 = null;
        EventRequest  eventRequest2 = null;

        String        property1     = "ClassPrepareRequest1";
        String        property2     = "ClassPrepareRequest2";

        Event newEvent = null;

        for (int i = 0; ; i++) {

            if (!shouldRunAfterBreakpoint()) {
                vm.resume();
                break;
            }

            display(":::::: case: # " + i);

            switch (i) {

                case 0:
                eventRequest1 = setting22ClassPrepareRequest(testedClassName1,
                                         EventRequest.SUSPEND_ALL, property1);

                display("......waiting for ClassPrepareEvent in expected thread");
                newEvent = eventHandler.waitForRequestedEvent(new EventRequest[]{eventRequest1}, waitTime, true);

                if ( !(newEvent instanceof ClassPrepareEvent)) {
                    setFailedStatus("ERROR: new event is not ClassPrepareEvent");
                } else {
                    String str = ((ClassPrepareEvent)newEvent).referenceType().name();
                    if (!str.endsWith(className1)) {
                        setFailedStatus("Received ClassPrepareEvent for unexpected class: \n\t" + str);
                    } else {
                        display("Received ClassPrepareEvent for expected class: \n\t" + str);
                    }

                    String property = (String) newEvent.request().getProperty("number");
                    display("       got new ClassPrepareEvent with property 'number' == " + property);

                    if ( !property.equals(property1) ) {
                        setFailedStatus("ERROR: property is not : " + property1);
                    }
                }

                vm.resume();
                break;

                case 1:
                eventRequest2 = setting22ClassPrepareRequest(testedClassName2,
                                         EventRequest.SUSPEND_ALL, property2);

                display("......waiting for ClassPrepareEvent in expected thread");
                newEvent = eventHandler.waitForRequestedEvent(new EventRequest[]{eventRequest2}, waitTime, true);

                if ( !(newEvent instanceof ClassPrepareEvent)) {
                    setFailedStatus("ERROR: new event is not ClassPrepareEvent");
                } else {

                    String str = ((ClassPrepareEvent)newEvent).referenceType().name();
                    if (!str.endsWith(className2)) {
                        setFailedStatus("Received ClassPrepareEvent for unexpected class: \n\t" + str);
                    } else {
                        display("Received ClassPrepareEvent for expected class: \n\t" + str);
                    }

                    String property = (String) newEvent.request().getProperty("number");
                    display("       got new ClassPrepareEvent with property 'number' == " + property);

                    if ( !property.equals(property2) ) {
                        setFailedStatus("ERROR: property is not : " + property2);
                    }
                }
                vm.resume();
                break;

                default:
                throw new Failure("** default case 2 **");
            }
        }
        return;
    }

    private ClassPrepareRequest setting22ClassPrepareRequest ( String testedClass,
                                                               int    suspendPolicy,
                                                               String property       ) {
        try {
            display("......setting up ClassPrepareRequest:");
            display("       class: " + testedClass + "; property: " + property);

            ClassPrepareRequest
            cpr = eventRManager.createClassPrepareRequest();
            cpr.putProperty("number", property);
            cpr.addClassFilter(testedClass);
            cpr.setSuspendPolicy(suspendPolicy);

            display("      ClassPrepareRequest has been set up");
            return cpr;
        } catch ( Exception e ) {
            throw new Failure("** FAILURE to set up ClassPrepareRequest **");
        }
    }

}
