///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/ColorPropertyUI.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ColorPropertyUI, PropertyParameterUI)

/******************************************************************************
* The constructor.
******************************************************************************/
ColorPropertyUI::ColorPropertyUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField)
	: PropertyParameterUI(parentEditor, propField)
{
	_label = new QLabel(propField.displayName() + ":");
	_colorPicker = new ColorPickerWidget();
	_colorPicker->setObjectName("colorButton");
	connect(_colorPicker, SIGNAL(colorChanged()), this, SLOT(onColorPickerChanged()));
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
ColorPropertyUI::~ColorPropertyUI()
{
	// Release GUI controls. 
	delete label();
	delete colorPicker();
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to.
******************************************************************************/
void ColorPropertyUI::resetUI()
{
	PropertyParameterUI::resetUI();
	
	if(colorPicker())  {
		if(editObject()) {
			colorPicker()->setEnabled(isEnabled());
		}
		else {
			colorPicker()->setEnabled(false);
			colorPicker()->setColor(Color(1,1,1));
		}
	}
}

/******************************************************************************
* This method updates the displayed value of the parameter UI.
******************************************************************************/
void ColorPropertyUI::updateUI()
{
	if(editObject() && colorPicker()) {
		if(propertyField()) {
			QVariant currentValue = editObject()->getPropertyFieldValue(*propertyField());
			OVITO_ASSERT(currentValue.isValid());
			if(currentValue.canConvert<Color>()) {
				Color val = currentValue.value<Color>();
				colorPicker()->setColor(val);
			}
		}
	}
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void ColorPropertyUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	PropertyParameterUI::setEnabled(enabled);
	if(colorPicker()) colorPicker()->setEnabled(editObject() != NULL && isEnabled());
}

/******************************************************************************
* Is called when the user has changed the color.
******************************************************************************/
void ColorPropertyUI::onColorPickerChanged()
{
	if(colorPicker() && editObject()) {
		if(propertyField()) {
			ViewportSuspender noVPUpdate;
			UNDO_MANAGER.beginCompoundOperation(tr("Change color"));
			QVariant newValue;
			newValue.setValue(colorPicker()->color());
			editObject()->setPropertyFieldValue(*propertyField(), newValue);
			UNDO_MANAGER.endCompoundOperation();
		}
	}
}

};

