/* ====================================================================
 *
 *
 *   BSD LICENSE
 *
 *   Copyright(c) 2022-2023 Intel Corporation.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *     * Neither the name of Intel Corporation nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * ====================================================================
 */

/*****************************************************************************
 * @file test_bssl_rsa.c
 *
 * This file provides a QAT Engine test functions.
 *
 *****************************************************************************/

/* macros defined to allow use of the cpu get and set affinity functions */
#ifndef _GNU_SOURCE
# define _GNU_SOURCE
#endif

#ifndef __USE_GNU
# define __USE_GNU
#endif

/* Local Includes */
#include "qat_bssl.h"
#include "qat_utils.h"
#include "test_bssl_utils.h"
#include "test_bssl_rsa.h"

/* OpenSSL Includes */
#include <openssl/ssl.h>

/* original data */
static unsigned char data_2048[] = {
  0x11, 0x11, 0x82, 0x63, 0x33, 0x9b, 0x25, 0x14, 0xfd, 0x80, 0x30, 0xc2,
  0x57, 0xa0, 0x79, 0x98, 0xd3, 0x66, 0x9d, 0x0e, 0x59, 0x2c, 0xc9, 0xd4,
  0x79, 0x83, 0x7d, 0xc5, 0xf1, 0xe9, 0xda, 0xf1, 0xa2, 0x42, 0x1b, 0xb9,
  0xac, 0xd0, 0xef, 0xce, 0x9a, 0x72, 0x41, 0xd6, 0xd0, 0xfb, 0xcc, 0xbb,
  0x1d, 0x0d, 0x62, 0xac, 0x95, 0x1d, 0x67, 0xab, 0xe8, 0xd0, 0xff, 0xe0,
  0x40, 0xb1, 0x3c, 0x25, 0xbf, 0xa4, 0xc0, 0xea, 0xdc, 0x0c, 0xdc, 0x37,
  0x6a, 0x25, 0x7a, 0xd2, 0x03, 0xf6, 0x83, 0xf8, 0x99, 0x48, 0x55, 0x5e,
  0x78, 0x0f, 0x39, 0x9c, 0xe0, 0xf9, 0x9c, 0x18, 0xe2, 0xbb, 0xd3, 0xa4,
  0xe3, 0x4b, 0x4e, 0x1c, 0x45, 0x8e, 0x17, 0x42, 0xfc, 0x0c, 0x0c, 0x42,
  0xcd, 0x29, 0x6c, 0x4f, 0x62, 0xd6, 0x86, 0xd7, 0x2f, 0x88, 0x76, 0x18,
  0xf1, 0xc1, 0xab, 0xf0, 0x12, 0x4d, 0xbe, 0x32, 0x4e, 0xaa, 0xd1, 0xd4,
  0x87, 0x9d, 0x25, 0xe2, 0xad, 0xc8, 0x26, 0xdc, 0xf8, 0xe8, 0x0c, 0xeb,
  0x5d, 0x51, 0x8f, 0x84, 0x63, 0x28, 0x5e, 0xb5, 0x75, 0xfa, 0x99, 0x7b,
  0xa3, 0xb7, 0x63, 0x98, 0xae, 0x4d, 0x76, 0x8f, 0x12, 0xee, 0xce, 0xcd,
  0x35, 0xc9, 0x6b, 0xa5, 0x01, 0x99, 0x49, 0x38, 0xe0, 0x97, 0xac, 0x4b,
  0x23, 0x62, 0x0d, 0x18, 0x3d, 0x20, 0xfc, 0xc1, 0x0e, 0x41, 0x49, 0x9e,
  0x40, 0xd5, 0x76, 0x25, 0xbb, 0x55, 0x66, 0xf0, 0x79, 0xb2, 0x7c, 0x38,
  0xa4, 0x07, 0x83, 0x97, 0xf0, 0x5d, 0x42, 0x89, 0x0f, 0x56, 0x36, 0x2c,
  0xd5, 0x88, 0xed, 0x42, 0x7f, 0xfe, 0x82, 0x86, 0x65, 0xef, 0x5d, 0xc8,
  0x18, 0x6c, 0x88, 0x6c, 0x50, 0x1b, 0xe9, 0x62, 0x85, 0x99, 0x7e, 0x8d,
  0x3e, 0xac, 0x4b, 0x21, 0x92, 0xd0, 0x2a, 0xc5, 0x31, 0xad, 0xe9, 0xe8,
  0x19, 0xfe, 0x6b, 0xa1
};

/* Original Data for 3K rsa test*/
static unsigned char data_3072[] = {
  0x61 ,0x46 ,0xe0 ,0x3c ,0x38 ,0xc0 ,0x9b ,0xdf ,0x78 ,0x87 ,0xe3 ,0x5e ,0x08 ,0xa1 ,0x97 ,0x53,
  0x41 ,0x90 ,0x8e ,0x34 ,0x1c ,0x91 ,0x73 ,0xd0 ,0xe1 ,0x08 ,0x2b ,0xf0 ,0x7d ,0xd1 ,0xa6 ,0x34,
  0x84 ,0xc3 ,0xd9 ,0x53 ,0x2f ,0x8a ,0x6b ,0xb3 ,0xcc ,0x6f ,0x95 ,0xe5 ,0x93 ,0x55 ,0x78 ,0xe4,
  0xd6 ,0x3b ,0x84 ,0x95 ,0xbf ,0xf9 ,0x38 ,0x30 ,0x5d ,0xbf ,0x00 ,0xe8 ,0xcd ,0xf2 ,0xfc ,0xbc,
  0x38 ,0xe3 ,0x25 ,0x1b ,0x3f ,0xc6 ,0x6d ,0x52 ,0x4c ,0x58 ,0x1d ,0xbe ,0x3e ,0x2b ,0x1e ,0x48,
  0x51 ,0xb5 ,0x38 ,0xed ,0x95 ,0xe9 ,0x1a ,0x07 ,0xfb ,0x2c ,0x29 ,0x0e ,0x87 ,0xa1 ,0xec ,0x12,
  0x2d ,0x75 ,0xfa ,0x99 ,0x76 ,0x0f ,0xd3 ,0x5b ,0xf8 ,0x10 ,0xeb ,0x52 ,0xf0 ,0x58 ,0x9c ,0x78,
  0xf7 ,0xb5 ,0xf7 ,0x84 ,0xc0 ,0xa1 ,0x6c ,0x99 ,0xce ,0x05 ,0xe3 ,0xaf ,0x60 ,0x92 ,0xc8 ,0x59,
  0x8b ,0x59 ,0xcf ,0x18 ,0xc5 ,0xfd ,0x57 ,0x2f ,0xd1 ,0x1c ,0x7b ,0x59 ,0x57 ,0x96 ,0x81 ,0xc8,
  0x7c ,0x67 ,0x24 ,0x8f ,0xdc ,0xc6 ,0x75 ,0x29 ,0x46 ,0xda ,0xab ,0x6c ,0x9c ,0xb4 ,0xce ,0xd9,
  0x18 ,0xb8 ,0xe3 ,0x9e ,0x83 ,0xb3 ,0x97 ,0x24 ,0x74 ,0x42 ,0x10 ,0x54 ,0xc6 ,0x54 ,0xac ,0xfa,
  0xcf ,0xbc ,0xe1 ,0x43 ,0xab ,0xcf ,0x38 ,0x94 ,0x69 ,0x52 ,0xdd ,0xb7 ,0xb1 ,0xdf ,0xb8 ,0xc9,
  0x9b ,0xcb ,0xe5 ,0x3f ,0x39 ,0x9c ,0x8f ,0x5d ,0x8e ,0x5f ,0xba ,0xc2 ,0x86 ,0x29 ,0xcd ,0x3e,
  0x76 ,0xae ,0x80 ,0x60 ,0xce ,0x1a ,0x7f ,0x33 ,0xef ,0x0a ,0x1e ,0x9f ,0xeb ,0xa9 ,0x1e ,0x43,
  0xb0 ,0xdf ,0x91 ,0xf4 ,0x3f ,0x1c ,0x84 ,0x6f ,0x43 ,0x57 ,0xc8 ,0xa6 ,0xd0 ,0x75 ,0x4e ,0xcc,
  0x34 ,0xaf ,0x36 ,0x08 ,0x23 ,0xef ,0x7d ,0x97 ,0x97 ,0xbf ,0x5d ,0x21 ,0xce ,0x14 ,0xa8 ,0xe3,
  0x4b ,0x5e ,0x4a ,0xe1 ,0x61 ,0xcf ,0x0a ,0xca ,0x37 ,0x5e ,0x00 ,0xc7 ,0x95 ,0xee ,0xc5 ,0x13,
  0xe5 ,0x4d ,0x9e ,0x73 ,0xdf ,0xd3 ,0x29 ,0xa6 ,0x53 ,0x43 ,0x64 ,0x50 ,0x26 ,0xbc ,0x2a ,0x68,
  0xb0 ,0x19 ,0x06 ,0xb1 ,0xdd ,0x20 ,0x72 ,0xd8 ,0xa7 ,0xd9 ,0x19 ,0x5b ,0x93 ,0xff ,0x01 ,0x0f,
  0x71 ,0x14 ,0xfa ,0x9f ,0xa5 ,0x92 ,0x37 ,0x31 ,0x8b ,0x6f ,0x3b ,0x04 ,0x64 ,0x8e ,0x14 ,0xb8,
  0x08 ,0x49 ,0xbe ,0x47 ,0x8a ,0x12 ,0xd8 ,0x42 ,0x93 ,0x20 ,0x54 ,0x20 ,0x8a ,0x3a ,0xe1 ,0xd5,
  0xfc ,0xf8 ,0x07 ,0x71 ,0x98 ,0x58 ,0x24 ,0xbc ,0x30 ,0x06 ,0xe2 ,0x8d ,0x7a ,0xd0 ,0xe9 ,0xc2,
  0x4a ,0x4e ,0x0e ,0xd2 ,0x3e ,0xdf ,0x5e ,0x1b ,0x2b ,0xdb ,0xfb ,0x19 ,0x10 ,0x07 ,0x33 ,0x72,
  0xe9 ,0x1c ,0x6c ,0x0f ,0x71 ,0x60 ,0x07 ,0x55 ,0x15 ,0x35 ,0x97 ,0x06 ,0x82 ,0x76 ,0xe9 ,0x10
};

/* Original Data for 4K rsa test*/
static unsigned char data_4096[] = {
  0x61 ,0x46 ,0xe0 ,0x3c ,0x38 ,0xc0 ,0x9b ,0xdf ,0x78 ,0x87 ,0xe3 ,0x5e ,0x08 ,0xa1 ,0x97 ,0x53,
  0x41 ,0x90 ,0x8e ,0x34 ,0x1c ,0x91 ,0x73 ,0xd0 ,0xe1 ,0x08 ,0x2b ,0xf0 ,0x7d ,0xd1 ,0xa6 ,0x34,
  0x84 ,0xc3 ,0xd9 ,0x53 ,0x2f ,0x8a ,0x6b ,0xb3 ,0xcc ,0x6f ,0x95 ,0xe5 ,0x93 ,0x55 ,0x78 ,0xe4,
  0xd6 ,0x3b ,0x84 ,0x95 ,0xbf ,0xf9 ,0x38 ,0x30 ,0x5d ,0xbf ,0x00 ,0xe8 ,0xcd ,0xf2 ,0xfc ,0xbc,
  0x38 ,0xe3 ,0x25 ,0x1b ,0x3f ,0xc6 ,0x6d ,0x52 ,0x4c ,0x58 ,0x1d ,0xbe ,0x3e ,0x2b ,0x1e ,0x48,
  0x51 ,0xb5 ,0x38 ,0xed ,0x95 ,0xe9 ,0x1a ,0x07 ,0xfb ,0x2c ,0x29 ,0x0e ,0x87 ,0xa1 ,0xec ,0x12,
  0x2d ,0x75 ,0xfa ,0x99 ,0x76 ,0x0f ,0xd3 ,0x5b ,0xf8 ,0x10 ,0xeb ,0x52 ,0xf0 ,0x58 ,0x9c ,0x78,
  0xf7 ,0xb5 ,0xf7 ,0x84 ,0xc0 ,0xa1 ,0x6c ,0x99 ,0xce ,0x05 ,0xe3 ,0xaf ,0x60 ,0x92 ,0xc8 ,0x59,
  0x8b ,0x59 ,0xcf ,0x18 ,0xc5 ,0xfd ,0x57 ,0x2f ,0xd1 ,0x1c ,0x7b ,0x59 ,0x57 ,0x96 ,0x81 ,0xc8,
  0x7c ,0x67 ,0x24 ,0x8f ,0xdc ,0xc6 ,0x75 ,0x29 ,0x46 ,0xda ,0xab ,0x6c ,0x9c ,0xb4 ,0xce ,0xd9,
  0x18 ,0xb8 ,0xe3 ,0x9e ,0x83 ,0xb3 ,0x97 ,0x24 ,0x74 ,0x42 ,0x10 ,0x54 ,0xc6 ,0x54 ,0xac ,0xfa,
  0xcf ,0xbc ,0xe1 ,0x43 ,0xab ,0xcf ,0x38 ,0x94 ,0x69 ,0x52 ,0xdd ,0xb7 ,0xb1 ,0xdf ,0xb8 ,0xc9,
  0x9b ,0xcb ,0xe5 ,0x3f ,0x39 ,0x9c ,0x8f ,0x5d ,0x8e ,0x5f ,0xba ,0xc2 ,0x86 ,0x29 ,0xcd ,0x3e,
  0x76 ,0xae ,0x80 ,0x60 ,0xce ,0x1a ,0x7f ,0x33 ,0xef ,0x0a ,0x1e ,0x9f ,0xeb ,0xa9 ,0x1e ,0x43,
  0xb0 ,0xdf ,0x91 ,0xf4 ,0x3f ,0x1c ,0x84 ,0x6f ,0x43 ,0x57 ,0xc8 ,0xa6 ,0xd0 ,0x75 ,0x4e ,0xcc,
  0x34 ,0xaf ,0x36 ,0x08 ,0x23 ,0xef ,0x7d ,0x97 ,0x97 ,0xbf ,0x5d ,0x21 ,0xce ,0x14 ,0xa8 ,0xe3,
  0x4b ,0x5e ,0x4a ,0xe1 ,0x61 ,0xcf ,0x0a ,0xca ,0x37 ,0x5e ,0x00 ,0xc7 ,0x95 ,0xee ,0xc5 ,0x13,
  0xe5 ,0x4d ,0x9e ,0x73 ,0xdf ,0xd3 ,0x29 ,0xa6 ,0x53 ,0x43 ,0x64 ,0x50 ,0x26 ,0xbc ,0x2a ,0x68,
  0xb0 ,0x19 ,0x06 ,0xb1 ,0xdd ,0x20 ,0x72 ,0xd8 ,0xa7 ,0xd9 ,0x19 ,0x5b ,0x93 ,0xff ,0x01 ,0x0f,
  0x71 ,0x14 ,0xfa ,0x9f ,0xa5 ,0x92 ,0x37 ,0x31 ,0x8b ,0x6f ,0x3b ,0x04 ,0x64 ,0x8e ,0x14 ,0xb8,
  0x08 ,0x49 ,0xbe ,0x47 ,0x8a ,0x12 ,0xd8 ,0x42 ,0x93 ,0x20 ,0x54 ,0x20 ,0x8a ,0x3a ,0xe1 ,0xd5,
  0xfc ,0xf8 ,0x07 ,0x71 ,0x98 ,0x58 ,0x24 ,0xbc ,0x30 ,0x06 ,0xe2 ,0x8d ,0x7a ,0xd0 ,0xe9 ,0xc2,
  0x4a ,0x4e ,0x0e ,0xd2 ,0x3e ,0xdf ,0x5e ,0x1b ,0x2b ,0xdb ,0xfb ,0x19 ,0x10 ,0x07 ,0x33 ,0x72,
  0xe9 ,0x1c ,0x6c ,0x0f ,0x71 ,0x60 ,0x07 ,0x55 ,0x15 ,0x35 ,0x97 ,0x06 ,0x82 ,0x76 ,0xe9 ,0x10,
  0x30 ,0x53 ,0x5a ,0xf8 ,0x5e ,0x2c ,0x3b ,0xd3 ,0xe7 ,0x07 ,0xc9 ,0xfc ,0x10 ,0x43 ,0xac ,0xb1,
  0xd7 ,0x03 ,0xdb ,0x08 ,0xa0 ,0x9e ,0x54 ,0xe2 ,0xa4 ,0x6e ,0x43 ,0xfe ,0xe4 ,0x6a ,0xe2 ,0xd2,
  0x8c ,0x0d ,0xb7 ,0xf7 ,0x75 ,0x5a ,0x2b ,0x68 ,0x99 ,0x97 ,0xf7 ,0xf9 ,0x22 ,0xa3 ,0x8a ,0xdc,
  0x67 ,0xf2 ,0x62 ,0x80 ,0x03 ,0xd6 ,0x4b ,0x41 ,0xe4 ,0xdf ,0x05 ,0x9b ,0x64 ,0xb0 ,0xaf ,0x73,
  0x4b ,0x54 ,0x42 ,0xdd ,0x95 ,0x01 ,0x46 ,0xa8 ,0xb2 ,0x90 ,0xb4 ,0xf1 ,0xf4 ,0x74 ,0x79 ,0x13,
  0x18 ,0xe5 ,0xcb ,0xe7 ,0xde ,0x48 ,0x2a ,0xbe ,0xf6 ,0xbe ,0x13 ,0x13 ,0x67 ,0x17 ,0xd9 ,0x74,
  0x38 ,0x07 ,0x76 ,0xa4 ,0x8f ,0x9a ,0x79 ,0xdb ,0x63 ,0x68 ,0x65 ,0xa6 ,0x8b ,0xa0 ,0x6c ,0xa9,
  0x35 ,0xf3 ,0xe4 ,0xa1 ,0xb9 ,0x37 ,0xdf ,0x1c ,0x92 ,0x2d ,0xc5 ,0x7b ,0xfd ,0xef ,0xa9 ,0x17
};

static pthread_t async_poll_thread;
static uint8_t *out_data = NULL;
static size_t out_len = 0;
static size_t max_len = 0;

static int qat_rsa_sign_test(RSA_METHOD *meth, RSA *rsa, const uint8_t *in_data,
                                size_t in_len, int async_mode);
static int qat_rsa_decrypt_test(RSA_METHOD *meth, RSA *rsa, const uint8_t *in_data,
                                size_t in_len, int async_mode);

static void qat_rsa_handle_async_ctx(async_ctx *ctx)
{
    if (NULL == out_data) {
        T_ERROR("out_data is NULL\n");
        goto err;
    }
    /* Copy data from async_ctx */
    if (bssl_qat_async_ctx_copy_result(ctx, out_data, &out_len, max_len)) {
        T_ERROR("Fail to get output results\n");
        goto err;
    }

err:
    bssl_qat_async_finish_job(ctx);
}

static void *qat_rsa_polling_async_ctx(void *args)
{
    async_ctx *ctx = (async_ctx *)args;
    T_DEBUG("RSA test polling result thread start\n");
    while (*ctx->currjob_status != ASYNC_JOB_COMPLETE) {
        usleep(10);
    }
    T_DEBUG("RSA test polling result thread detect data ready\n");

    qat_rsa_handle_async_ctx(ctx);
    T_DEBUG("RSA test polling result thread finished\n");

    return (void*)0;
}

static async_ctx *qat_rsa_init_async_ctx(void) {

    async_ctx *ctx = bssl_qat_async_start_job();
    ASYNC_JOB *job = ctx->currjob;
    T_DEBUG("RSA init async_ctx async ctx:%p, job:%p, job->status:%d \n",
             ctx, job, job->status);
    pthread_create(&async_poll_thread, NULL, qat_rsa_polling_async_ctx, ctx);

    return ctx;
}

static void qat_rsa_wait_async_ctx(size_t *out_len)
{
    pthread_join(async_poll_thread, NULL);
}

/* async_mode 0: sync mode 1: async mode */
int qat_rsa_test(const EVP_PKEY *pkey, int flag)
{
    RSA *rsa;
    RSA_METHOD *meth;
    size_t rsa_size;
    uint8_t *in_data;

    rsa = EVP_PKEY_get0_RSA(pkey);
    rsa_size = RSA_size(rsa); /* bytes */
    switch(rsa_size) {
        case 256:
            in_data = data_2048;
            break;
        case 384:
            in_data = data_3072;
            break;
        case 512:
            in_data = data_4096;
            break;
        default:
            T_WARN("RSA size %ld bits not supported, 2k/3k/4k suggested\n",
                rsa_size * 8);
            return -1;
    }

    if (NULL == (meth = bssl_engine_get_rsa_method())) {
        return -1;
    }

    if (flag & RSA_ASYNC_MODE) {
        max_len = rsa_size;
        qat_rsa_init_async_ctx();
    }

    if (flag & RSA_DECRYPT_TEST) {
        qat_rsa_decrypt_test(meth, rsa, in_data, rsa_size, flag & RSA_ASYNC_MODE);
    } else {
        qat_rsa_sign_test(meth, rsa, in_data, rsa_size, flag & RSA_ASYNC_MODE);
    }

    return out_len;
}

int qat_rsa_sign_test(RSA_METHOD *meth, RSA *rsa, const uint8_t *in_data,
                      size_t in_len, int async_mode)
{

    uint8_t *ver_data = NULL;

    if (!meth || !rsa) {
        return 1; /* error */
    }

    if (NULL == (out_data = (unsigned char *)OPENSSL_zalloc(in_len))) {
        T_ERROR("Failed to allocate buffer\n");
        return 1; /* error */
    }

    /* Signing */
    T_DUMP_RSA_SIGN_INPUT(in_data, in_len);
    meth->sign_raw(rsa, &out_len, out_data, in_len, in_data, in_len,
                RSA_NO_PADDING);

    if (async_mode) {
        qat_rsa_wait_async_ctx(&out_len);
    }

    T_DUMP_RSA_SIGN_OUTPUT(out_data, out_len);
    if (in_len != out_len) {
        T_ERROR("RSA Sign: wrong length\n");
        return 1; /* fail */
    } else {
        T_DEBUG("RSA Sign(%s mode): OK\n", async_mode?"Async":"Sync");
    }

    /* Verifying */
    if (NULL == (ver_data = (unsigned char *)OPENSSL_zalloc(in_len))) {
        T_ERROR("Failed to allocate buffer\n");
        return 1; /* fail */
    }

    out_len = RSA_meth_get_pub_dec(RSA_PKCS1_OpenSSL())(in_len, out_data, ver_data,
        rsa, RSA_NO_PADDING);
    if (in_len != out_len) {
        T_ERROR("RSA Verify: wrong length\n");
    } else {
        T_DEBUG("RSA Verify: OK\n");
    }

    OPENSSL_free(ver_data);
    return 0;
}

int qat_rsa_decrypt_test(RSA_METHOD *meth, RSA *rsa, const uint8_t *in_data,
                            size_t in_len, int async_mode)
{
    uint8_t *enc_data = NULL; /* for public encryption */

    if (!meth || !rsa) {
        return 1; /* error */
    }

    if (NULL == (enc_data = (unsigned char *)OPENSSL_zalloc(in_len))) {
        T_ERROR("Failed to allocate buffer\n");
        return 1; /* fail */
    }

    /* Pub encrypt */
    out_len = RSA_meth_get_pub_enc(RSA_PKCS1_OpenSSL())(in_len, in_data, enc_data,
                                   rsa, RSA_NO_PADDING);
    if (in_len != out_len) {
        T_ERROR("RSA pub encrypt: wrong length\n");
    } else {
        T_DEBUG("RSA pub encrypt: OK\n");
    }

     /* Decrypting */
    if (NULL == (out_data = (unsigned char *)OPENSSL_zalloc(in_len))) {
        T_ERROR("Failed to allocate buffer\n");
        OPENSSL_free(enc_data);
        return 1; /* error */
    }
    T_DUMP_RSA_DECRYPT_INPUT(enc_data, out_len);
    meth->decrypt(rsa, &out_len, out_data, in_len, enc_data, in_len,
                RSA_NO_PADDING);

    if (async_mode) {
        qat_rsa_wait_async_ctx(&out_len);
    }

    T_DUMP_RSA_DECRYPT_OUTPUT(out_data, out_len);
    if (in_len != out_len) {
        T_ERROR("RSA Decrypt: wrong length\n");
        OPENSSL_free(enc_data);
        OPENSSL_free(out_data);
        return 1; /* fail */
    } else {
        T_DEBUG("RSA Decrypt(%s mode): OK\n", async_mode?"Async":"Sync");
    }


    OPENSSL_free(enc_data);
    return 0;
}
